<?php

namespace App\Http\Controllers;

use App\Http\Requests\ResendEmailRequest;
use App\Http\Requests\ValidateRegisterRequest;
use App\Mail\VerifyMail;
use App\Models\User;
use Illuminate\Http\Request;

use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;


class RegisterController extends Controller
{
     /**
      * Register in the app.
      *
      * @OA\Post(
      *     path="/api/register",
      *     tags={"Регистрация пользователя, заполнение формы"},
      *     summary="Запись нового пользователя в БД",
      *     operationId="registerUser",
      *     @OA\Parameter(
      *             name="name",
      *             in="query",
      *             required=true,
      *             @OA\Schema(type="string"),
      *             description="Введение ФИО пользователем"
      *         ),
      *    @OA\Parameter(
      *             name="email",
      *             in="query",
      *             required=true,
      *             @OA\Schema(type="string"),
      *             description="Введение email пользователем"
      *         ),
      *    @OA\Parameter(
      *             name="tel",
      *             in="query",
      *             required=true,
      *             @OA\Schema(type="integer"),
      *             description="Номер телефона пользователя"
      *         ),
      *     @OA\Parameter(
      *             name="password",
      *             in="query",
      *             required=true,
      *             @OA\Schema(type="string"),
      *             description="Пароль пользователя"
      *         ),
      *      @OA\Parameter(
      *             name="password_confirmed",
      *             in="query",
      *             required=true,
      *             @OA\Schema(type="string"),
      *             description="Пароль пользователя"
      *         ),
      *     @OA\Response(
      *         response="200",
      *         description="Ok",
      *         @OA\JsonContent(
      *             @OA\Property(property="message", type="string", example="Пользователь успешно зарегистрирован"),
      *             @OA\Property(property="status", type="boolean", example=true)
      *         ),
      *     ),
      *     @OA\RequestBody(
      *         required=true,
      *         @OA\JsonContent(ref="#/components/schemas/ValidateRegisterRequest")
      *     )
      * )
      **/

    public function store(ValidateRegisterRequest $request)
    {
        $user = User::create([
            'name' => $request ['name'],
            'tel' => $request ['tel'],
            'email' => $request['email'],
            'password' => Hash::make($request['password']),
            'city' => $request['city'],
            'verify_token' => Str::random(),
            'status' => User::STATUS_INACTIVE,
        ]);
        Mail::to($user->email)->send(new VerifyMail($user));

        return response()->json([
            'message' => "Пользователь успешно зарегистрирован",
            'status' => true
        ]);
    }

    /**
     *     Resend the email to the previously entered email address.
     *
     *     @OA\Post(
     *     path="/api/register-mail-again",
     *     tags={"Повторная отправка подтверждения регистрации пользователю"},
     *     summary="Повторная отправка эл.письма пользователю",
     *     operationId="registerMailAgain",
     *     @OA\Parameter(
     *             name="email",
     *             in="cookie",
     *             required=true,
     *             @OA\Schema(type="string", example="panama@mail.ru"),
     *             description="Поле не заполняется пользователем, получаем из объекта запроса"
     *         ),
     *     @OA\Response(
     *         response="200",
     *         description="Ok",
     *         @OA\JsonContent(
     *             @OA\Property(property="message", type="string", example="Письмо отправлено"),
     *             @OA\Property(property="status", type="boolean", example=true)
     *         ),
     *     ),
     * )
     **/

    public function again(ResendEmailRequest $request) {
        if(!$user = User::where('email', $request->email)) {
            return response()->json([
                'error' => 'Failed',
                'status' => false
            ]);
        }
        $user = $user->firstOrFail();
        Mail::to([$user->email])->send(new VerifyMail($user));
        return response()->json([
            'message' => "Письмо отправлено",
            'status' => true
        ]);
    }

    /**
     *     Email verification, automatic filling of the token field.
     *
     *     @OA\Get(
     *     path="/api/verify",
     *     tags={"Подтверждение записи о пользователе"},
     *     summary="Верификация email пользователя и внесение изменений в БД",
     *     operationId="verifyUser",
     *      @OA\Parameter(
     *             name="_token",
     *             in="cookie",
     *             required=true,
     *             @OA\Schema(type="string", example="fPMgGFTHFGTZgRfFSr"),
     *             description="Поле не заполняется пользователем, получаем из объекта запроса"
     *         ),
     *      @OA\Response(
     *         response="200",
     *         description="Ok",
     *         @OA\JsonContent(
     *             @OA\Property(property="message", type="string", example="Email подтвержден"),
     *             @OA\Property(property="status", type="boolean", example=true)
     *         ),
     *     ),
     * )
     **/

    public function verify(Request $request)
    {
        if (!$user = User::where('verify_token', $request['token'])->first()) {
            return response()->json([
                'error' => 'Verification failed',
                'message' => "Пользователь с таким ID не найден",
                'status' => false
            ]);
        }
        $user->email_verified_at = Carbon::now();
        $user->status = User::STATUS_ACTIVE;
        $user->verify_token = null;
        $user->save();
        return response()->json([
            'message' => "Email подтвержден",
            'status' => true
        ]);
    }

    public function city() {
        $cities =[
            [ "id" => 1, "name" => "Абаза"],
            [ "id" => 2, "name" => "Абакан"],
            [ "id" => 3, "name" => "Абдулино"],
            [ "id" => 4, "name" => "Абинск"],
            [ "id" => 5, "name" => "Агидель"],
            [ "id" => 6, "name" => "Агрыз"],
            [ "id" => 7, "name" => "Адыгейск"],
            [ "id" => 8, "name" => "Азнакаево"],
            [ "id" => 9, "name" => "Азов"],
            [ "id" => 10, "name" => "Ак-Довурак"],
            [ "id" => 11, "name" => "Аксай"],
            [ "id" => 12, "name" => "Алагир"],
            [ "id" => 13, "name" => "Алапаевск"],
            [ "id" => 14, "name" => "Алатырь"],
            [ "id" => 15, "name" => "Алдан"],
            [ "id" => 16, "name" => "Алейск"],
            [ "id" => 17, "name" => "Александров"],
            [ "id" => 18, "name" => "Александровск"],
            [ "id" => 19, "name" => "Александровск-Сахалинский"],
            [ "id" => 20, "name" => "Алексеевка"],
            [ "id" => 21, "name" => "Алексин"],
            [ "id" => 22, "name" => "Алзамай"],
            [ "id" => 23, "name" => "Алупка"],
            [ "id" => 24, "name" => "Алушта"],
            [ "id" => 25, "name" => "Альметьевск"],
            [ "id" => 26, "name" => "Амурск"],
            [ "id" => 27, "name" => "Анадырь"],
            [ "id" => 28, "name" => "Анапа"],
            [ "id" => 29, "name" => "Ангарск"],
            [ "id" => 30, "name" => "Андреаполь"],
            [ "id" => 31, "name" => "Анжеро-Судженск"],
            [ "id" => 32, "name" => "Анива"],
            [ "id" => 33, "name" => "Апатиты"],
            [ "id" => 34, "name" => "Апрелевка"],
            [ "id" => 35, "name" => "Апшеронск"],
            [ "id" => 36, "name" => "Арамиль"],
            [ "id" => 37, "name" => "Аргун"],
            [ "id" => 38, "name" => "Ардатов"],
            [ "id" => 39, "name" => "Ардон"],
            [ "id" => 40, "name" => "Арзамас"],
            [ "id" => 41, "name" => "Аркадак"],
            [ "id" => 42, "name" => "Армавир"],
            [ "id" => 43, "name" => "Армянск"],
            [ "id" => 44, "name" => "Арсеньев"],
            [ "id" => 45, "name" => "Арск"],
            [ "id" => 46, "name" => "Артём"],
            [ "id" => 47, "name" => "Артёмовск"],
            [ "id" => 48, "name" => "Артёмовский"],
            [ "id" => 49, "name" => "Архангельск"],
            [ "id" => 50, "name" => "Асбест"],
            [ "id" => 51, "name" => "Асино"],
            [ "id" => 52, "name" => "Астрахань"],
            [ "id" => 53, "name" => "Аткарск"],
            [ "id" => 54, "name" => "Ахтубинск"],
            [ "id" => 55, "name" => "Ачинск"],
            [ "id" => 56, "name" => "Аша"],
            [ "id" => 57, "name" => "Бабаево"],
            [ "id" => 58, "name" => "Бабушкин"],
            [ "id" => 59, "name" => "Бавлы"],
            [ "id" => 60, "name" => "Багратионовск"],
            [ "id" => 61, "name" => "Байкальск"],
            [ "id" => 62, "name" => "Баймак"],
            [ "id" => 63, "name" => "Бакал"],
            [ "id" => 64, "name" => "Баксан"],
            [ "id" => 65, "name" => "Балабаново"],
            [ "id" => 66, "name" => "Балаково"],
            [ "id" => 67, "name" => "Балахна"],
            [ "id" => 68, "name" => "Балашиха"],
            [ "id" => 69, "name" => "Балашов"],
            [ "id" => 70, "name" => "Балей"],
            [ "id" => 71, "name" => "Балтийск"],
            [ "id" => 72, "name" => "Барабинск"],
            [ "id" => 73, "name" => "Барнаул"],
            [ "id" => 74, "name" => "Барыш"],
            [ "id" => 75, "name" => "Батайск"],
            [ "id" => 76, "name" => "Бахчисарай"],
            [ "id" => 77, "name" => "Бежецк"],
            [ "id" => 78, "name" => "Белая Калитва"],
            [ "id" => 79, "name" => "Белая Холуница"],
            [ "id" => 80, "name" => "Белгород"],
            [ "id" => 81, "name" => "Белебей"],
            [ "id" => 82, "name" => "Белёв"],
            [ "id" => 83, "name" => "Белинский"],
            [ "id" => 84, "name" => "Белово"],
            [ "id" => 85, "name" => "Белогорск"],
            [ "id" => 86, "name" => "Белогорск"],
            [ "id" => 87, "name" => "Белозерск"],
            [ "id" => 88, "name" => "Белокуриха"],
            [ "id" => 89, "name" => "Беломорск"],
            [ "id" => 90, "name" => "Белоозёрский"],
            [ "id" => 91, "name" => "Белорецк"],
            [ "id" => 92, "name" => "Белореченск"],
            [ "id" => 93, "name" => "Белоусово"],
            [ "id" => 94, "name" => "Белоярский"],
            [ "id" => 95, "name" => "Белый"],
            [ "id" => 96, "name" => "Бердск"],
            [ "id" => 97, "name" => "Березники"],
            [ "id" => 98, "name" => "Берёзовский"],
            [ "id" => 99, "name" => "Берёзовский"],
            [ "id" => 100, "name" => "Беслан"],
            [ "id" => 101, "name" => "Бийск"],
            [ "id" => 102, "name" => "Бикин"],
            [ "id" => 103, "name" => "Билибино"],
            [ "id" => 104, "name" => "Биробиджан"],
            [ "id" => 105, "name" => "Бирск"],
            [ "id" => 106, "name" => "Бирюсинск"],
            [ "id" => 107, "name" => "Бирюч"],
            [ "id" => 108, "name" => "Благовещенск"],
            [ "id" => 109, "name" => "Благовещенск"],
            [ "id" => 110, "name" => "Благодарный"],
            [ "id" => 111, "name" => "Бобров"],
            [ "id" => 112, "name" => "Богданович"],
            [ "id" => 113, "name" => "Богородицк"],
            [ "id" => 114, "name" => "Богородск"],
            [ "id" => 115, "name" => "Боготол"],
            [ "id" => 116, "name" => "Богучар"],
            [ "id" => 117, "name" => "Бодайбо"],
            [ "id" => 118, "name" => "Бокситогорск"],
            [ "id" => 119, "name" => "Болгар"],
            [ "id" => 120, "name" => "Бологое"],
            [ "id" => 121, "name" => "Болотное"],
            [ "id" => 122, "name" => "Болохово"],
            [ "id" => 123, "name" => "Болхов"],
            [ "id" => 124, "name" => "Большой Камень"],
            [ "id" => 125, "name" => "Бор"],
            [ "id" => 126, "name" => "Борзя"],
            [ "id" => 127, "name" => "Борисоглебск"],
            [ "id" => 128, "name" => "Боровичи"],
            [ "id" => 129, "name" => "Боровск"],
            [ "id" => 130, "name" => "Бородино"],
            [ "id" => 131, "name" => "Братск"],
            [ "id" => 132, "name" => "Бронницы"],
            [ "id" => 133, "name" => "Брянск"],
            [ "id" => 134, "name" => "Бугульма"],
            [ "id" => 135, "name" => "Бугуруслан"],
            [ "id" => 136, "name" => "Будённовск"],
            [ "id" => 137, "name" => "Бузулук"],
            [ "id" => 138, "name" => "Буинск"],
            [ "id" => 139, "name" => "Буй"],
            [ "id" => 140, "name" => "Буйнакск"],
            [ "id" => 141, "name" => "Бутурлиновка"],
            [ "id" => 142, "name" => "Валдай"],
            [ "id" => 143, "name" => "Валуйки"],
            [ "id" => 144, "name" => "Велиж"],
            [ "id" => 145, "name" => "Великие Луки"],
            [ "id" => 146, "name" => "Великий Новгород"],
            [ "id" => 147, "name" => "Великий Устюг"],
            [ "id" => 148, "name" => "Вельск"],
            [ "id" => 149, "name" => "Венёв"],
            [ "id" => 150, "name" => "Верещагино"],
            [ "id" => 151, "name" => "Верея"],
            [ "id" => 152, "name" => "Верхнеуральск"],
            [ "id" => 153, "name" => "Верхний Тагил"],
            [ "id" => 154, "name" => "Верхний Уфалей"],
            [ "id" => 155, "name" => "Верхняя Пышма"],
            [ "id" => 156, "name" => "Верхняя Салда"],
            [ "id" => 157, "name" => "Верхняя Тура"],
            [ "id" => 158, "name" => "Верхотурье"],
            [ "id" => 159, "name" => "Верхоянск"],
            [ "id" => 160, "name" => "Весьегонск"],
            [ "id" => 161, "name" => "Ветлуга"],
            [ "id" => 162, "name" => "Видное"],
            [ "id" => 163, "name" => "Вилюйск"],
            [ "id" => 164, "name" => "Вилючинск"],
            [ "id" => 165, "name" => "Вихоревка"],
            [ "id" => 166, "name" => "Вичуга"],
            [ "id" => 167, "name" => "Владивосток"],
            [ "id" => 168, "name" => "Владикавказ"],
            [ "id" => 169, "name" => "Владимир"],
            [ "id" => 170, "name" => "Волгоград"],
            [ "id" => 171, "name" => "Волгодонск"],
            [ "id" => 172, "name" => "Волгореченск"],
            [ "id" => 173, "name" => "Волжск"],
            [ "id" => 174, "name" => "Волжский"],
            [ "id" => 175, "name" => "Вологда"],
            [ "id" => 176, "name" => "Володарск"],
            [ "id" => 177, "name" => "Волоколамск"],
            [ "id" => 178, "name" => "Волосово"],
            [ "id" => 179, "name" => "Волхов"],
            [ "id" => 180, "name" => "Волчанск"],
            [ "id" => 181, "name" => "Вольск"],
            [ "id" => 182, "name" => "Воркута"],
            [ "id" => 183, "name" => "Воронеж"],
            [ "id" => 184, "name" => "Ворсма"],
            [ "id" => 185, "name" => "Воскресенск"],
            [ "id" => 186, "name" => "Воткинск"],
            [ "id" => 187, "name" => "Всеволожск"],
            [ "id" => 188, "name" => "Вуктыл"],
            [ "id" => 189, "name" => "Выборг"],
            [ "id" => 190, "name" => "Выкса"],
            [ "id" => 191, "name" => "Высоковск"],
            [ "id" => 192, "name" => "Высоцк"],
            [ "id" => 193, "name" => "Вытегра"],
            [ "id" => 194, "name" => "Вышний Волочёк"],
            [ "id" => 195, "name" => "Вяземский"],
            [ "id" => 196, "name" => "Вязники"],
            [ "id" => 197, "name" => "Вязьма"],
            [ "id" => 198, "name" => "Вятские Поляны"],
            [ "id" => 199, "name" => "Гаврилов Посад"],
            [ "id" => 200, "name" => "Гаврилов-Ям"],
            [ "id" => 201, "name" => "Гагарин"],
            [ "id" => 202, "name" => "Гаджиево"],
            [ "id" => 203, "name" => "Гай"],
            [ "id" => 204, "name" => "Галич"],
            [ "id" => 205, "name" => "Гатчина"],
            [ "id" => 206, "name" => "Гвардейск"],
            [ "id" => 207, "name" => "Гдов"],
            [ "id" => 208, "name" => "Геленджик"],
            [ "id" => 209, "name" => "Георгиевск"],
            [ "id" => 210, "name" => "Глазов"],
            [ "id" => 211, "name" => "Голицыно"],
            [ "id" => 212, "name" => "Горбатов"],
            [ "id" => 213, "name" => "Горно-Алтайск"],
            [ "id" => 214, "name" => "Горнозаводск"],
            [ "id" => 215, "name" => "Горняк"],
            [ "id" => 216, "name" => "Городец"],
            [ "id" => 217, "name" => "Городище"],
            [ "id" => 218, "name" => "Городовиковск"],
            [ "id" => 219, "name" => "Гороховец"],
            [ "id" => 220, "name" => "Горячий Ключ"],
            [ "id" => 221, "name" => "Грайворон"],
            [ "id" => 222, "name" => "Гремячинск"],
            [ "id" => 223, "name" => "Грозный"],
            [ "id" => 224, "name" => "Грязи"],
            [ "id" => 225, "name" => "Грязовец"],
            [ "id" => 226, "name" => "Губаха"],
            [ "id" => 227, "name" => "Губкин"],
            [ "id" => 228, "name" => "Губкинский"],
            [ "id" => 229, "name" => "Гудермес"],
            [ "id" => 230, "name" => "Гуково"],
            [ "id" => 231, "name" => "Гулькевичи"],
            [ "id" => 232, "name" => "Гурьевск"],
            [ "id" => 233, "name" => "Гурьевск"],
            [ "id" => 234, "name" => "Гусев"],
            [ "id" => 235, "name" => "Гусиноозёрск"],
            [ "id" => 236, "name" => "Гусь-Хрустальный"],
            [ "id" => 237, "name" => "Давлеканово"],
            [ "id" => 238, "name" => "Дагестанские Огни"],
            [ "id" => 239, "name" => "Далматово"],
            [ "id" => 240, "name" => "Дальнегорск"],
            [ "id" => 241, "name" => "Дальнереченск"],
            [ "id" => 242, "name" => "Данилов"],
            [ "id" => 243, "name" => "Данков"],
            [ "id" => 244, "name" => "Дегтярск"],
            [ "id" => 245, "name" => "Дедовск"],
            [ "id" => 246, "name" => "Демидов"],
            [ "id" => 247, "name" => "Дербент"],
            [ "id" => 248, "name" => "Десногорск"],
            [ "id" => 249, "name" => "Джанкой"],
            [ "id" => 250, "name" => "Дзержинск"],
            [ "id" => 251, "name" => "Дзержинский"],
            [ "id" => 252, "name" => "Дивногорск"],
            [ "id" => 253, "name" => "Дигора"],
            [ "id" => 254, "name" => "Димитровград"],
            [ "id" => 255, "name" => "Дмитриев"],
            [ "id" => 256, "name" => "Дмитров"],
            [ "id" => 257, "name" => "Дмитровск"],
            [ "id" => 258, "name" => "Дно"],
            [ "id" => 259, "name" => "Добрянка"],
            [ "id" => 260, "name" => "Долгопрудный"],
            [ "id" => 261, "name" => "Долинск"],
            [ "id" => 262, "name" => "Домодедово"],
            [ "id" => 263, "name" => "Донецк"],
            [ "id" => 264, "name" => "Донской"],
            [ "id" => 265, "name" => "Дорогобуж"],
            [ "id" => 266, "name" => "Дрезна"],
            [ "id" => 267, "name" => "Дубна"],
            [ "id" => 268, "name" => "Дубовка"],
            [ "id" => 269, "name" => "Дудинка"],
            [ "id" => 270, "name" => "Духовщина"],
            [ "id" => 271, "name" => "Дюртюли"],
            [ "id" => 272, "name" => "Дятьково"],
            [ "id" => 273, "name" => "Евпатория"],
            [ "id" => 274, "name" => "Егорьевск"],
            [ "id" => 275, "name" => "Ейск"],
            [ "id" => 276, "name" => "Екатеринбург"],
            [ "id" => 277, "name" => "Елабуга"],
            [ "id" => 278, "name" => "Елец"],
            [ "id" => 279, "name" => "Елизово"],
            [ "id" => 280, "name" => "Ельня"],
            [ "id" => 281, "name" => "Еманжелинск"],
            [ "id" => 282, "name" => "Емва"],
            [ "id" => 283, "name" => "Енисейск"],
            [ "id" => 284, "name" => "Ермолино"],
            [ "id" => 285, "name" => "Ершов"],
            [ "id" => 286, "name" => "Ессентуки"],
            [ "id" => 287, "name" => "Ефремов"],
            [ "id" => 288, "name" => "Железноводск"],
            [ "id" => 289, "name" => "Железногорск"],
            [ "id" => 290, "name" => "Железногорск"],
            [ "id" => 291, "name" => "Железногорск-Илимский"],
            [ "id" => 292, "name" => "Жердевка"],
            [ "id" => 293, "name" => "Жигулёвск"],
            [ "id" => 294, "name" => "Жиздра"],
            [ "id" => 295, "name" => "Жирновск"],
            [ "id" => 296, "name" => "Жуков"],
            [ "id" => 297, "name" => "Жуковка"],
            [ "id" => 298, "name" => "Жуковский"],
            [ "id" => 299, "name" => "Завитинск"],
            [ "id" => 300, "name" => "Заводоуковск"],
            [ "id" => 301, "name" => "Заволжск"],
            [ "id" => 302, "name" => "Заволжье"],
            [ "id" => 303, "name" => "Задонск"],
            [ "id" => 304, "name" => "Заинск"],
            [ "id" => 305, "name" => "Закаменск"],
            [ "id" => 306, "name" => "Заозёрный"],
            [ "id" => 307, "name" => "Заозёрск"],
            [ "id" => 308, "name" => "Западная Двина"],
            [ "id" => 309, "name" => "Заполярный"],
            [ "id" => 310, "name" => "Зарайск"],
            [ "id" => 311, "name" => "Заречный"],
            [ "id" => 312, "name" => "Заречный"],
            [ "id" => 313, "name" => "Заринск"],
            [ "id" => 314, "name" => "Звенигово"],
            [ "id" => 315, "name" => "Звенигород"],
            [ "id" => 316, "name" => "Зверево"],
            [ "id" => 317, "name" => "Зеленогорск"],
            [ "id" => 318, "name" => "Зеленоградск"],
            [ "id" => 319, "name" => "Зеленодольск"],
            [ "id" => 320, "name" => "Зеленокумск"],
            [ "id" => 321, "name" => "Зерноград"],
            [ "id" => 322, "name" => "Зея"],
            [ "id" => 323, "name" => "Зима"],
            [ "id" => 324, "name" => "Златоуст"],
            [ "id" => 325, "name" => "Злынка"],
            [ "id" => 326, "name" => "Змеиногорск"],
            [ "id" => 327, "name" => "Знаменск"],
            [ "id" => 328, "name" => "Зубцов"],
            [ "id" => 329, "name" => "Зуевка"],
            [ "id" => 330, "name" => "Ивангород"],
            [ "id" => 331, "name" => "Иваново"],
            [ "id" => 332, "name" => "Ивантеевка"],
            [ "id" => 333, "name" => "Ивдель"],
            [ "id" => 334, "name" => "Игарка"],
            [ "id" => 335, "name" => "Ижевск"],
            [ "id" => 336, "name" => "Избербаш"],
            [ "id" => 337, "name" => "Изобильный"],
            [ "id" => 338, "name" => "Иланский"],
            [ "id" => 339, "name" => "Инза"],
            [ "id" => 340, "name" => "Иннополис"],
            [ "id" => 341, "name" => "Инсар"],
            [ "id" => 342, "name" => "Инта"],
            [ "id" => 343, "name" => "Ипатово"],
            [ "id" => 344, "name" => "Ирбит"],
            [ "id" => 345, "name" => "Иркутск"],
            [ "id" => 346, "name" => "Исилькуль"],
            [ "id" => 347, "name" => "Искитим"],
            [ "id" => 348, "name" => "Истра"],
            [ "id" => 349, "name" => "Ишим"],
            [ "id" => 350, "name" => "Ишимбай"],
            [ "id" => 351, "name" => "Йошкар-Ола"],
            [ "id" => 352, "name" => "Кадников"],
            [ "id" => 353, "name" => "Казань"],
            [ "id" => 354, "name" => "Калач"],
            [ "id" => 355, "name" => "Калач-на-Дону"],
            [ "id" => 356, "name" => "Калачинск"],
            [ "id" => 357, "name" => "Калининград"],
            [ "id" => 358, "name" => "Калининск"],
            [ "id" => 359, "name" => "Калтан"],
            [ "id" => 360, "name" => "Калуга"],
            [ "id" => 361, "name" => "Калязин"],
            [ "id" => 362, "name" => "Камбарка"],
            [ "id" => 363, "name" => "Каменка"],
            [ "id" => 364, "name" => "Каменногорск"],
            [ "id" => 365, "name" => "Каменск-Уральский"],
            [ "id" => 366, "name" => "Каменск-Шахтинский"],
            [ "id" => 367, "name" => "Камень-на-Оби"],
            [ "id" => 368, "name" => "Камешково"],
            [ "id" => 369, "name" => "Камызяк"],
            [ "id" => 370, "name" => "Камышин"],
            [ "id" => 371, "name" => "Камышлов"],
            [ "id" => 372, "name" => "Канаш"],
            [ "id" => 373, "name" => "Кандалакша"],
            [ "id" => 374, "name" => "Канск"],
            [ "id" => 375, "name" => "Карабаново"],
            [ "id" => 376, "name" => "Карабаш"],
            [ "id" => 377, "name" => "Карабулак"],
            [ "id" => 378, "name" => "Карасук"],
            [ "id" => 379, "name" => "Карачаевск"],
            [ "id" => 380, "name" => "Карачев"],
            [ "id" => 381, "name" => "Каргат"],
            [ "id" => 382, "name" => "Каргополь"],
            [ "id" => 383, "name" => "Карпинск"],
            [ "id" => 384, "name" => "Карталы"],
            [ "id" => 385, "name" => "Касимов"],
            [ "id" => 386, "name" => "Касли"],
            [ "id" => 387, "name" => "Каспийск"],
            [ "id" => 388, "name" => "Катав-Ивановск"],
            [ "id" => 389, "name" => "Катайск"],
            [ "id" => 390, "name" => "Качканар"],
            [ "id" => 391, "name" => "Кашин"],
            [ "id" => 392, "name" => "Кашира"],
            [ "id" => 393, "name" => "Кедровый"],
            [ "id" => 394, "name" => "Кемерово"],
            [ "id" => 395, "name" => "Кемь"],
            [ "id" => 396, "name" => "Керчь"],
            [ "id" => 397, "name" => "Кизел"],
            [ "id" => 398, "name" => "Кизилюрт"],
            [ "id" => 399, "name" => "Кизляр"],
            [ "id" => 400, "name" => "Кимовск"],
            [ "id" => 401, "name" => "Кимры"],
            [ "id" => 402, "name" => "Кингисепп"],
            [ "id" => 403, "name" => "Кинель"],
            [ "id" => 404, "name" => "Кинешма"],
            [ "id" => 405, "name" => "Киреевск"],
            [ "id" => 406, "name" => "Киренск"],
            [ "id" => 407, "name" => "Киржач"],
            [ "id" => 408, "name" => "Кириллов"],
            [ "id" => 409, "name" => "Кириши"],
            [ "id" => 410, "name" => "Киров"],
            [ "id" => 411, "name" => "Киров"],
            [ "id" => 412, "name" => "Кировград"],
            [ "id" => 413, "name" => "Кирово-Чепецк"],
            [ "id" => 414, "name" => "Кировск"],
            [ "id" => 415, "name" => "Кировск"],
            [ "id" => 416, "name" => "Кирс"],
            [ "id" => 417, "name" => "Кирсанов"],
            [ "id" => 418, "name" => "Киселёвск"],
            [ "id" => 419, "name" => "Кисловодск"],
            [ "id" => 420, "name" => "Клин"],
            [ "id" => 421, "name" => "Клинцы"],
            [ "id" => 422, "name" => "Княгинино"],
            [ "id" => 423, "name" => "Ковдор"],
            [ "id" => 424, "name" => "Ковров"],
            [ "id" => 425, "name" => "Ковылкино"],
            [ "id" => 426, "name" => "Когалым"],
            [ "id" => 427, "name" => "Кодинск"],
            [ "id" => 428, "name" => "Козельск"],
            [ "id" => 429, "name" => "Козловка"],
            [ "id" => 430, "name" => "Козьмодемьянск"],
            [ "id" => 431, "name" => "Кола"],
            [ "id" => 432, "name" => "Кологрив"],
            [ "id" => 433, "name" => "Коломна"],
            [ "id" => 434, "name" => "Колпашево"],
            [ "id" => 435, "name" => "Кольчугино"],
            [ "id" => 436, "name" => "Коммунар"],
            [ "id" => 437, "name" => "Комсомольск"],
            [ "id" => 438, "name" => "Комсомольск-на-Амуре"],
            [ "id" => 439, "name" => "Конаково"],
            [ "id" => 440, "name" => "Кондопога"],
            [ "id" => 441, "name" => "Кондрово"],
            [ "id" => 442, "name" => "Константиновск"],
            [ "id" => 443, "name" => "Копейск"],
            [ "id" => 444, "name" => "Кораблино"],
            [ "id" => 445, "name" => "Кореновск"],
            [ "id" => 446, "name" => "Коркино"],
            [ "id" => 447, "name" => "Королёв"],
            [ "id" => 448, "name" => "Короча"],
            [ "id" => 449, "name" => "Корсаков"],
            [ "id" => 450, "name" => "Коряжма"],
            [ "id" => 451, "name" => "Костерёво"],
            [ "id" => 452, "name" => "Костомукша"],
            [ "id" => 453, "name" => "Кострома"],
            [ "id" => 454, "name" => "Котельники"],
            [ "id" => 455, "name" => "Котельниково"],
            [ "id" => 456, "name" => "Котельнич"],
            [ "id" => 457, "name" => "Котлас"],
            [ "id" => 458, "name" => "Котово"],
            [ "id" => 459, "name" => "Котовск"],
            [ "id" => 460, "name" => "Кохма"],
            [ "id" => 461, "name" => "Красавино"],
            [ "id" => 462, "name" => "Красноармейск"],
            [ "id" => 463, "name" => "Красноармейск"],
            [ "id" => 464, "name" => "Красновишерск"],
            [ "id" => 465, "name" => "Красногорск"],
            [ "id" => 466, "name" => "Краснодар"],
            [ "id" => 467, "name" => "Краснозаводск"],
            [ "id" => 468, "name" => "Краснознаменск"],
            [ "id" => 469, "name" => "Краснознаменск"],
            [ "id" => 470, "name" => "Краснокаменск"],
            [ "id" => 471, "name" => "Краснокамск"],
            [ "id" => 472, "name" => "Красноперекопск"],
            [ "id" => 473, "name" => "Краснослободск"],
            [ "id" => 474, "name" => "Краснослободск"],
            [ "id" => 475, "name" => "Краснотурьинск"],
            [ "id" => 476, "name" => "Красноуральск"],
            [ "id" => 477, "name" => "Красноуфимск"],
            [ "id" => 478, "name" => "Красноярск"],
            [ "id" => 479, "name" => "Красный Кут"],
            [ "id" => 480, "name" => "Красный Сулин"],
            [ "id" => 481, "name" => "Красный Холм"],
            [ "id" => 482, "name" => "Кремёнки"],
            [ "id" => 483, "name" => "Кропоткин"],
            [ "id" => 484, "name" => "Крымск"],
            [ "id" => 485, "name" => "Кстово"],
            [ "id" => 486, "name" => "Кубинка"],
            [ "id" => 487, "name" => "Кувандык"],
            [ "id" => 488, "name" => "Кувшиново"],
            [ "id" => 489, "name" => "Кудрово"],
            [ "id" => 490, "name" => "Кудымкар"],
            [ "id" => 491, "name" => "Кузнецк"],
            [ "id" => 492, "name" => "Куйбышев"],
            [ "id" => 493, "name" => "Кукмор"],
            [ "id" => 494, "name" => "Кулебаки"],
            [ "id" => 495, "name" => "Кумертау"],
            [ "id" => 496, "name" => "Кунгур"],
            [ "id" => 497, "name" => "Купино"],
            [ "id" => 498, "name" => "Курган"],
            [ "id" => 499, "name" => "Курганинск"],
            [ "id" => 500, "name" => "Курильск"],
            [ "id" => 501, "name" => "Курлово"],
            [ "id" => 502, "name" => "Куровское"],
            [ "id" => 503, "name" => "Курск"],
            [ "id" => 504, "name" => "Куртамыш"],
            [ "id" => 505, "name" => "Курчалой"],
            [ "id" => 506, "name" => "Курчатов"],
            [ "id" => 507, "name" => "Куса"],
            [ "id" => 508, "name" => "Кушва"],
            [ "id" => 509, "name" => "Кызыл"],
            [ "id" => 510, "name" => "Кыштым"],
            [ "id" => 511, "name" => "Кяхта"],
            [ "id" => 512, "name" => "Лабинск"],
            [ "id" => 513, "name" => "Лабытнанги"],
            [ "id" => 514, "name" => "Лагань"],
            [ "id" => 515, "name" => "Ладушкин"],
            [ "id" => 516, "name" => "Лаишево"],
            [ "id" => 517, "name" => "Лакинск"],
            [ "id" => 518, "name" => "Лангепас"],
            [ "id" => 519, "name" => "Лахденпохья"],
            [ "id" => 520, "name" => "Лебедянь"],
            [ "id" => 521, "name" => "Лениногорск"],
            [ "id" => 522, "name" => "Ленинск"],
            [ "id" => 523, "name" => "Ленинск-Кузнецкий"],
            [ "id" => 524, "name" => "Ленск"],
            [ "id" => 525, "name" => "Лермонтов"],
            [ "id" => 526, "name" => "Лесной"],
            [ "id" => 527, "name" => "Лесозаводск"],
            [ "id" => 528, "name" => "Лесосибирск"],
            [ "id" => 529, "name" => "Ливны"],
            [ "id" => 530, "name" => "Ликино-Дулёво"],
            [ "id" => 531, "name" => "Липецк"],
            [ "id" => 532, "name" => "Липки"],
            [ "id" => 533, "name" => "Лиски"],
            [ "id" => 534, "name" => "Лихославль"],
            [ "id" => 535, "name" => "Лобня"],
            [ "id" => 536, "name" => "Лодейное Поле"],
            [ "id" => 537, "name" => "Лосино-Петровский"],
            [ "id" => 538, "name" => "Луга"],
            [ "id" => 539, "name" => "Луза"],
            [ "id" => 540, "name" => "Лукоянов"],
            [ "id" => 541, "name" => "Луховицы"],
            [ "id" => 542, "name" => "Лысково"],
            [ "id" => 543, "name" => "Лысьва"],
            [ "id" => 544, "name" => "Лыткарино"],
            [ "id" => 545, "name" => "Льгов"],
            [ "id" => 546, "name" => "Любань"],
            [ "id" => 547, "name" => "Люберцы"],
            [ "id" => 548, "name" => "Любим"],
            [ "id" => 549, "name" => "Людиново"],
            [ "id" => 550, "name" => "Лянтор"],
            [ "id" => 551, "name" => "Магадан"],
            [ "id" => 552, "name" => "Магас"],
            [ "id" => 553, "name" => "Магнитогорск"],
            [ "id" => 554, "name" => "Майкоп"],
            [ "id" => 555, "name" => "Майский"],
            [ "id" => 556, "name" => "Макаров"],
            [ "id" => 557, "name" => "Макарьев"],
            [ "id" => 558, "name" => "Макушино"],
            [ "id" => 559, "name" => "Малая Вишера"],
            [ "id" => 560, "name" => "Малгобек"],
            [ "id" => 561, "name" => "Малмыж"],
            [ "id" => 562, "name" => "Малоархангельск"],
            [ "id" => 563, "name" => "Малоярославец"],
            [ "id" => 564, "name" => "Мамадыш"],
            [ "id" => 565, "name" => "Мамоново"],
            [ "id" => 566, "name" => "Мантурово"],
            [ "id" => 567, "name" => "Мариинск"],
            [ "id" => 568, "name" => "Мариинский Посад"],
            [ "id" => 569, "name" => "Маркс"],
            [ "id" => 570, "name" => "Махачкала"],
            [ "id" => 571, "name" => "Мглин"],
            [ "id" => 572, "name" => "Мегион"],
            [ "id" => 573, "name" => "Медвежьегорск"],
            [ "id" => 574, "name" => "Медногорск"],
            [ "id" => 575, "name" => "Медынь"],
            [ "id" => 576, "name" => "Межгорье"],
            [ "id" => 577, "name" => "Междуреченск"],
            [ "id" => 578, "name" => "Мезень"],
            [ "id" => 579, "name" => "Меленки"],
            [ "id" => 580, "name" => "Мелеуз"],
            [ "id" => 581, "name" => "Менделеевск"],
            [ "id" => 582, "name" => "Мензелинск"],
            [ "id" => 583, "name" => "Мещовск"],
            [ "id" => 584, "name" => "Миасс"],
            [ "id" => 585, "name" => "Микунь"],
            [ "id" => 586, "name" => "Миллерово"],
            [ "id" => 587, "name" => "Минеральные Воды"],
            [ "id" => 588, "name" => "Минусинск"],
            [ "id" => 589, "name" => "Миньяр"],
            [ "id" => 590, "name" => "Мирный"],
            [ "id" => 591, "name" => "Мирный"],
            [ "id" => 592, "name" => "Михайлов"],
            [ "id" => 593, "name" => "Михайловка"],
            [ "id" => 594, "name" => "Михайловск"],
            [ "id" => 595, "name" => "Михайловск"],
            [ "id" => 596, "name" => "Мичуринск"],
            [ "id" => 597, "name" => "Могоча"],
            [ "id" => 598, "name" => "Можайск"],
            [ "id" => 599, "name" => "Можга"],
            [ "id" => 600, "name" => "Моздок"],
            [ "id" => 601, "name" => "Мончегорск"],
            [ "id" => 602, "name" => "Морозовск"],
            [ "id" => 603, "name" => "Моршанск"],
            [ "id" => 604, "name" => "Мосальск"],
            [ "id" => 605, "name" => "Москва"],
            [ "id" => 606, "name" => "Муравленко"],
            [ "id" => 607, "name" => "Мураши"],
            [ "id" => 608, "name" => "Мурино"],
            [ "id" => 609, "name" => "Мурманск"],
            [ "id" => 610, "name" => "Муром"],
            [ "id" => 611, "name" => "Мценск"],
            [ "id" => 612, "name" => "Мыски"],
            [ "id" => 613, "name" => "Мытищи"],
            [ "id" => 614, "name" => "Мышкин"],
            [ "id" => 615, "name" => "Набережные Челны"],
            [ "id" => 616, "name" => "Навашино"],
            [ "id" => 617, "name" => "Наволоки"],
            [ "id" => 618, "name" => "Надым"],
            [ "id" => 619, "name" => "Назарово"],
            [ "id" => 620, "name" => "Назрань"],
            [ "id" => 621, "name" => "Называевск"],
            [ "id" => 622, "name" => "Нальчик"],
            [ "id" => 623, "name" => "Нариманов"],
            [ "id" => 624, "name" => "Наро-Фоминск"],
            [ "id" => 625, "name" => "Нарткала"],
            [ "id" => 626, "name" => "Нарьян-Мар"],
            [ "id" => 627, "name" => "Находка"],
            [ "id" => 628, "name" => "Невель"],
            [ "id" => 629, "name" => "Невельск"],
            [ "id" => 630, "name" => "Невинномысск"],
            [ "id" => 631, "name" => "Невьянск"],
            [ "id" => 632, "name" => "Нелидово"],
            [ "id" => 633, "name" => "Неман"],
            [ "id" => 634, "name" => "Нерехта"],
            [ "id" => 635, "name" => "Нерчинск"],
            [ "id" => 636, "name" => "Нерюнгри"],
            [ "id" => 637, "name" => "Нестеров"],
            [ "id" => 638, "name" => "Нефтегорск"],
            [ "id" => 639, "name" => "Нефтекамск"],
            [ "id" => 640, "name" => "Нефтекумск"],
            [ "id" => 641, "name" => "Нефтеюганск"],
            [ "id" => 642, "name" => "Нея"],
            [ "id" => 643, "name" => "Нижневартовск"],
            [ "id" => 644, "name" => "Нижнекамск"],
            [ "id" => 645, "name" => "Нижнеудинск"],
            [ "id" => 646, "name" => "Нижние Серги"],
            [ "id" => 647, "name" => "Нижний Ломов"],
            [ "id" => 648, "name" => "Нижний Новгород"],
            [ "id" => 649, "name" => "Нижний Тагил"],
            [ "id" => 650, "name" => "Нижняя Салда"],
            [ "id" => 651, "name" => "Нижняя Тура"],
            [ "id" => 652, "name" => "Николаевск"],
            [ "id" => 653, "name" => "Николаевск-на-Амуре"],
            [ "id" => 654, "name" => "Никольск"],
            [ "id" => 655, "name" => "Никольск"],
            [ "id" => 656, "name" => "Никольское"],
            [ "id" => 657, "name" => "Новая Ладога"],
            [ "id" => 658, "name" => "Новая Ляля"],
            [ "id" => 659, "name" => "Новоалександровск"],
            [ "id" => 660, "name" => "Новоалтайск"],
            [ "id" => 661, "name" => "Новоаннинский"],
            [ "id" => 662, "name" => "Нововоронеж"],
            [ "id" => 663, "name" => "Новодвинск"],
            [ "id" => 664, "name" => "Новозыбков"],
            [ "id" => 665, "name" => "Новокубанск"],
            [ "id" => 666, "name" => "Новокузнецк"],
            [ "id" => 667, "name" => "Новокуйбышевск"],
            [ "id" => 668, "name" => "Новомичуринск"],
            [ "id" => 669, "name" => "Новомосковск"],
            [ "id" => 670, "name" => "Новопавловск"],
            [ "id" => 671, "name" => "Новоржев"],
            [ "id" => 672, "name" => "Новороссийск"],
            [ "id" => 673, "name" => "Новосибирск"],
            [ "id" => 674, "name" => "Новосиль"],
            [ "id" => 675, "name" => "Новосокольники"],
            [ "id" => 676, "name" => "Новотроицк"],
            [ "id" => 677, "name" => "Новоузенск"],
            [ "id" => 678, "name" => "Новоульяновск"],
            [ "id" => 679, "name" => "Новоуральск"],
            [ "id" => 680, "name" => "Новохопёрск"],
            [ "id" => 681, "name" => "Новочебоксарск"],
            [ "id" => 682, "name" => "Новочеркасск"],
            [ "id" => 683, "name" => "Новошахтинск"],
            [ "id" => 684, "name" => "Новый Оскол"],
            [ "id" => 685, "name" => "Новый Уренгой"],
            [ "id" => 686, "name" => "Ногинск"],
            [ "id" => 687, "name" => "Нолинск"],
            [ "id" => 688, "name" => "Норильск"],
            [ "id" => 689, "name" => "Ноябрьск"],
            [ "id" => 690, "name" => "Нурлат"],
            [ "id" => 691, "name" => "Нытва"],
            [ "id" => 692, "name" => "Нюрба"],
            [ "id" => 693, "name" => "Нягань"],
            [ "id" => 694, "name" => "Нязепетровск"],
            [ "id" => 695, "name" => "Няндома"],
            [ "id" => 696, "name" => "Облучье"],
            [ "id" => 697, "name" => "Обнинск"],
            [ "id" => 698, "name" => "Обоянь"],
            [ "id" => 699, "name" => "Обь"],
            [ "id" => 700, "name" => "Одинцово"],
            [ "id" => 701, "name" => "Озёрск"],
            [ "id" => 702, "name" => "Озёрск"],
            [ "id" => 703, "name" => "Озёры"],
            [ "id" => 704, "name" => "Октябрьск"],
            [ "id" => 705, "name" => "Октябрьский"],
            [ "id" => 706, "name" => "Окуловка"],
            [ "id" => 707, "name" => "Олёкминск"],
            [ "id" => 708, "name" => "Оленегорск"],
            [ "id" => 709, "name" => "Олонец"],
            [ "id" => 710, "name" => "Омск"],
            [ "id" => 711, "name" => "Омутнинск"],
            [ "id" => 712, "name" => "Онега"],
            [ "id" => 713, "name" => "Опочка"],
            [ "id" => 714, "name" => "Орёл"],
            [ "id" => 715, "name" => "Оренбург"],
            [ "id" => 716, "name" => "Орехово-Зуево"],
            [ "id" => 717, "name" => "Орлов"],
            [ "id" => 718, "name" => "Орск"],
            [ "id" => 719, "name" => "Оса"],
            [ "id" => 720, "name" => "Осинники"],
            [ "id" => 721, "name" => "Осташков"],
            [ "id" => 722, "name" => "Остров"],
            [ "id" => 723, "name" => "Островной"],
            [ "id" => 724, "name" => "Острогожск"],
            [ "id" => 725, "name" => "Отрадное"],
            [ "id" => 726, "name" => "Отрадный"],
            [ "id" => 727, "name" => "Оха"],
            [ "id" => 728, "name" => "Оханск"],
            [ "id" => 729, "name" => "Очёр"],
            [ "id" => 730, "name" => "Павлово"],
            [ "id" => 731, "name" => "Павловск"],
            [ "id" => 732, "name" => "Павловский Посад"],
            [ "id" => 733, "name" => "Палласовка"],
            [ "id" => 734, "name" => "Партизанск"],
            [ "id" => 735, "name" => "Певек"],
            [ "id" => 736, "name" => "Пенза"],
            [ "id" => 737, "name" => "Первомайск"],
            [ "id" => 738, "name" => "Первоуральск"],
            [ "id" => 739, "name" => "Перевоз"],
            [ "id" => 740, "name" => "Пересвет"],
            [ "id" => 741, "name" => "Переславль-Залесский"],
            [ "id" => 742, "name" => "Пермь"],
            [ "id" => 743, "name" => "Пестово"],
            [ "id" => 744, "name" => "Петров Вал"],
            [ "id" => 745, "name" => "Петровск"],
            [ "id" => 746, "name" => "Петровск-Забайкальский"],
            [ "id" => 747, "name" => "Петрозаводск"],
            [ "id" => 748, "name" => "Петропавловск-Камчатский"],
            [ "id" => 749, "name" => "Петухово"],
            [ "id" => 750, "name" => "Петушки"],
            [ "id" => 751, "name" => "Печора"],
            [ "id" => 752, "name" => "Печоры"],
            [ "id" => 753, "name" => "Пикалёво"],
            [ "id" => 754, "name" => "Пионерский"],
            [ "id" => 755, "name" => "Питкяранта"],
            [ "id" => 756, "name" => "Плавск"],
            [ "id" => 757, "name" => "Пласт"],
            [ "id" => 758, "name" => "Плёс"],
            [ "id" => 759, "name" => "Поворино"],
            [ "id" => 760, "name" => "Подольск"],
            [ "id" => 761, "name" => "Подпорожье"],
            [ "id" => 762, "name" => "Покачи"],
            [ "id" => 763, "name" => "Покров"],
            [ "id" => 764, "name" => "Покровск"],
            [ "id" => 765, "name" => "Полевской"],
            [ "id" => 766, "name" => "Полесск"],
            [ "id" => 767, "name" => "Полысаево"],
            [ "id" => 768, "name" => "Полярные Зори"],
            [ "id" => 769, "name" => "Полярный"],
            [ "id" => 770, "name" => "Поронайск"],
            [ "id" => 771, "name" => "Порхов"],
            [ "id" => 772, "name" => "Похвистнево"],
            [ "id" => 773, "name" => "Почеп"],
            [ "id" => 774, "name" => "Починок"],
            [ "id" => 775, "name" => "Пошехонье"],
            [ "id" => 776, "name" => "Правдинск"],
            [ "id" => 777, "name" => "Приволжск"],
            [ "id" => 778, "name" => "Приморск"],
            [ "id" => 779, "name" => "Приморск"],
            [ "id" => 780, "name" => "Приморско-Ахтарск"],
            [ "id" => 781, "name" => "Приозерск"],
            [ "id" => 782, "name" => "Прокопьевск"],
            [ "id" => 783, "name" => "Пролетарск"],
            [ "id" => 784, "name" => "Протвино"],
            [ "id" => 785, "name" => "Прохладный"],
            [ "id" => 786, "name" => "Псков"],
            [ "id" => 787, "name" => "Пугачёв"],
            [ "id" => 788, "name" => "Пудож"],
            [ "id" => 789, "name" => "Пустошка"],
            [ "id" => 790, "name" => "Пучеж"],
            [ "id" => 791, "name" => "Пушкино"],
            [ "id" => 792, "name" => "Пущино"],
            [ "id" => 793, "name" => "Пыталово"],
            [ "id" => 794, "name" => "Пыть-Ях"],
            [ "id" => 795, "name" => "Пятигорск"],
            [ "id" => 796, "name" => "Радужный"],
            [ "id" => 797, "name" => "Радужный"],
            [ "id" => 798, "name" => "Райчихинск"],
            [ "id" => 799, "name" => "Раменское"],
            [ "id" => 800, "name" => "Рассказово"],
            [ "id" => 801, "name" => "Ревда"],
            [ "id" => 802, "name" => "Реж"],
            [ "id" => 803, "name" => "Реутов"],
            [ "id" => 804, "name" => "Ржев"],
            [ "id" => 805, "name" => "Родники"],
            [ "id" => 806, "name" => "Рославль"],
            [ "id" => 807, "name" => "Россошь"],
            [ "id" => 808, "name" => "Ростов"],
            [ "id" => 809, "name" => "Ростов-на-Дону"],
            [ "id" => 810, "name" => "Рошаль"],
            [ "id" => 811, "name" => "Ртищево"],
            [ "id" => 812, "name" => "Рубцовск"],
            [ "id" => 813, "name" => "Рудня"],
            [ "id" => 814, "name" => "Руза"],
            [ "id" => 815, "name" => "Рузаевка"],
            [ "id" => 816, "name" => "Рыбинск"],
            [ "id" => 817, "name" => "Рыбное"],
            [ "id" => 818, "name" => "Рыльск"],
            [ "id" => 819, "name" => "Ряжск"],
            [ "id" => 820, "name" => "Рязань"],
            [ "id" => 821, "name" => "Саки"],
            [ "id" => 822, "name" => "Салават"],
            [ "id" => 823, "name" => "Салаир"],
            [ "id" => 824, "name" => "Салехард"],
            [ "id" => 825, "name" => "Сальск"],
            [ "id" => 826, "name" => "Самара"],
            [ "id" => 827, "name" => "Санкт-Петербург"],
            [ "id" => 828, "name" => "Саранск"],
            [ "id" => 829, "name" => "Сарапул"],
            [ "id" => 830, "name" => "Саратов"],
            [ "id" => 831, "name" => "Саров"],
            [ "id" => 832, "name" => "Сасово"],
            [ "id" => 833, "name" => "Сатка"],
            [ "id" => 834, "name" => "Сафоново"],
            [ "id" => 835, "name" => "Саяногорск"],
            [ "id" => 836, "name" => "Саянск"],
            [ "id" => 837, "name" => "Светлогорск"],
            [ "id" => 838, "name" => "Светлоград"],
            [ "id" => 839, "name" => "Светлый"],
            [ "id" => 840, "name" => "Светогорск"],
            [ "id" => 841, "name" => "Свирск"],
            [ "id" => 842, "name" => "Свободный"],
            [ "id" => 843, "name" => "Себеж"],
            [ "id" => 844, "name" => "Севастополь"],
            [ "id" => 845, "name" => "Северо-Курильск"],
            [ "id" => 846, "name" => "Северобайкальск"],
            [ "id" => 847, "name" => "Северодвинск"],
            [ "id" => 848, "name" => "Североморск"],
            [ "id" => 849, "name" => "Североуральск"],
            [ "id" => 850, "name" => "Северск"],
            [ "id" => 851, "name" => "Севск"],
            [ "id" => 852, "name" => "Сегежа"],
            [ "id" => 853, "name" => "Сельцо"],
            [ "id" => 854, "name" => "Семёнов"],
            [ "id" => 855, "name" => "Семикаракорск"],
            [ "id" => 856, "name" => "Семилуки"],
            [ "id" => 857, "name" => "Сенгилей"],
            [ "id" => 858, "name" => "Серафимович"],
            [ "id" => 859, "name" => "Сергач"],
            [ "id" => 860, "name" => "Сергиев Посад"],
            [ "id" => 861, "name" => "Сердобск"],
            [ "id" => 862, "name" => "Серов"],
            [ "id" => 863, "name" => "Серпухов"],
            [ "id" => 864, "name" => "Сертолово"],
            [ "id" => 865, "name" => "Сибай"],
            [ "id" => 866, "name" => "Сим"],
            [ "id" => 867, "name" => "Симферополь"],
            [ "id" => 868, "name" => "Сковородино"],
            [ "id" => 869, "name" => "Скопин"],
            [ "id" => 870, "name" => "Славгород"],
            [ "id" => 871, "name" => "Славск"],
            [ "id" => 872, "name" => "Славянск-на-Кубани"],
            [ "id" => 873, "name" => "Сланцы"],
            [ "id" => 874, "name" => "Слободской"],
            [ "id" => 875, "name" => "Слюдянка"],
            [ "id" => 876, "name" => "Смоленск"],
            [ "id" => 877, "name" => "Снежинск"],
            [ "id" => 878, "name" => "Снежногорск"],
            [ "id" => 879, "name" => "Собинка"],
            [ "id" => 880, "name" => "Советск"],
            [ "id" => 881, "name" => "Советск"],
            [ "id" => 882, "name" => "Советск"],
            [ "id" => 883, "name" => "Советская Гавань"],
            [ "id" => 884, "name" => "Советский"],
            [ "id" => 885, "name" => "Сокол"],
            [ "id" => 886, "name" => "Солигалич"],
            [ "id" => 887, "name" => "Соликамск"],
            [ "id" => 888, "name" => "Солнечногорск"],
            [ "id" => 889, "name" => "Соль-Илецк"],
            [ "id" => 890, "name" => "Сольвычегодск"],
            [ "id" => 891, "name" => "Сольцы"],
            [ "id" => 892, "name" => "Сорочинск"],
            [ "id" => 893, "name" => "Сорск"],
            [ "id" => 894, "name" => "Сортавала"],
            [ "id" => 895, "name" => "Сосенский"],
            [ "id" => 896, "name" => "Сосновка"],
            [ "id" => 897, "name" => "Сосновоборск"],
            [ "id" => 898, "name" => "Сосновый Бор"],
            [ "id" => 899, "name" => "Сосногорск"],
            [ "id" => 900, "name" => "Сочи"],
            [ "id" => 901, "name" => "Спас-Деменск"],
            [ "id" => 902, "name" => "Спас-Клепики"],
            [ "id" => 903, "name" => "Спасск"],
            [ "id" => 904, "name" => "Спасск-Дальний"],
            [ "id" => 905, "name" => "Спасск-Рязанский"],
            [ "id" => 906, "name" => "Среднеколымск"],
            [ "id" => 907, "name" => "Среднеуральск"],
            [ "id" => 908, "name" => "Сретенск"],
            [ "id" => 909, "name" => "Ставрополь"],
            [ "id" => 910, "name" => "Старая Купавна"],
            [ "id" => 911, "name" => "Старая Русса"],
            [ "id" => 912, "name" => "Старица"],
            [ "id" => 913, "name" => "Стародуб"],
            [ "id" => 914, "name" => "Старый Крым"],
            [ "id" => 915, "name" => "Старый Оскол"],
            [ "id" => 916, "name" => "Стерлитамак"],
            [ "id" => 917, "name" => "Стрежевой"],
            [ "id" => 918, "name" => "Строитель"],
            [ "id" => 919, "name" => "Струнино"],
            [ "id" => 920, "name" => "Ступино"],
            [ "id" => 921, "name" => "Суворов"],
            [ "id" => 922, "name" => "Судак"],
            [ "id" => 923, "name" => "Суджа"],
            [ "id" => 924, "name" => "Судогда"],
            [ "id" => 925, "name" => "Суздаль"],
            [ "id" => 926, "name" => "Сунжа"],
            [ "id" => 927, "name" => "Суоярви"],
            [ "id" => 928, "name" => "Сураж"],
            [ "id" => 929, "name" => "Сургут"],
            [ "id" => 930, "name" => "Суровикино"],
            [ "id" => 931, "name" => "Сурск"],
            [ "id" => 932, "name" => "Сусуман"],
            [ "id" => 933, "name" => "Сухиничи"],
            [ "id" => 934, "name" => "Сухой Лог"],
            [ "id" => 935, "name" => "Сызрань"],
            [ "id" => 936, "name" => "Сыктывкар"],
            [ "id" => 937, "name" => "Сысерть"],
            [ "id" => 938, "name" => "Сычёвка"],
            [ "id" => 939, "name" => "Сясьстрой"],
            [ "id" => 940, "name" => "Тавда"],
            [ "id" => 941, "name" => "Таганрог"],
            [ "id" => 942, "name" => "Тайга"],
            [ "id" => 943, "name" => "Тайшет"],
            [ "id" => 944, "name" => "Талдом"],
            [ "id" => 945, "name" => "Талица"],
            [ "id" => 946, "name" => "Тамбов"],
            [ "id" => 947, "name" => "Тара"],
            [ "id" => 948, "name" => "Тарко-Сале"],
            [ "id" => 949, "name" => "Таруса"],
            [ "id" => 950, "name" => "Татарск"],
            [ "id" => 951, "name" => "Таштагол"],
            [ "id" => 952, "name" => "Тверь"],
            [ "id" => 953, "name" => "Теберда"],
            [ "id" => 954, "name" => "Тейково"],
            [ "id" => 955, "name" => "Темников"],
            [ "id" => 956, "name" => "Темрюк"],
            [ "id" => 957, "name" => "Терек"],
            [ "id" => 958, "name" => "Тетюши"],
            [ "id" => 959, "name" => "Тимашёвск"],
            [ "id" => 960, "name" => "Тихвин"],
            [ "id" => 961, "name" => "Тихорецк"],
            [ "id" => 962, "name" => "Тобольск"],
            [ "id" => 963, "name" => "Тогучин"],
            [ "id" => 964, "name" => "Тольятти"],
            [ "id" => 965, "name" => "Томари"],
            [ "id" => 966, "name" => "Томмот"],
            [ "id" => 967, "name" => "Томск"],
            [ "id" => 968, "name" => "Топки"],
            [ "id" => 969, "name" => "Торжок"],
            [ "id" => 970, "name" => "Торопец"],
            [ "id" => 971, "name" => "Тосно"],
            [ "id" => 972, "name" => "Тотьма"],
            [ "id" => 973, "name" => "Трёхгорный"],
            [ "id" => 974, "name" => "Троицк"],
            [ "id" => 975, "name" => "Трубчевск"],
            [ "id" => 976, "name" => "Туапсе"],
            [ "id" => 977, "name" => "Туймазы"],
            [ "id" => 978, "name" => "Тула"],
            [ "id" => 979, "name" => "Тулун"],
            [ "id" => 980, "name" => "Туран"],
            [ "id" => 981, "name" => "Туринск"],
            [ "id" => 982, "name" => "Тутаев"],
            [ "id" => 983, "name" => "Тында"],
            [ "id" => 984, "name" => "Тырныауз"],
            [ "id" => 985, "name" => "Тюкалинск"],
            [ "id" => 986, "name" => "Тюмень"],
            [ "id" => 987, "name" => "Уварово"],
            [ "id" => 988, "name" => "Углегорск"],
            [ "id" => 989, "name" => "Углич"],
            [ "id" => 990, "name" => "Удачный"],
            [ "id" => 991, "name" => "Удомля"],
            [ "id" => 992, "name" => "Ужур"],
            [ "id" => 993, "name" => "Узловая"],
            [ "id" => 994, "name" => "Улан-Удэ"],
            [ "id" => 995, "name" => "Ульяновск"],
            [ "id" => 996, "name" => "Унеча"],
            [ "id" => 997, "name" => "Урай"],
            [ "id" => 998, "name" => "Урень"],
            [ "id" => 999, "name" => "Уржум"],
            [ "id" => 1000, "name" => "Урус-Мартан"],
            [ "id" => 1001, "name" => "Урюпинск"],
            [ "id" => 1002, "name" => "Усинск"],
            [ "id" => 1003, "name" => "Усмань"],
            [ "id" => 1004, "name" => "Усолье"],
            [ "id" => 1005, "name" => "Усолье-Сибирское"],
            [ "id" => 1006, "name" => "Уссурийск"],
            [ "id" => 1007, "name" => "Усть-Джегута"],
            [ "id" => 1008, "name" => "Усть-Илимск"],
            [ "id" => 1009, "name" => "Усть-Катав"],
            [ "id" => 1010, "name" => "Усть-Кут"],
            [ "id" => 1011, "name" => "Усть-Лабинск"],
            [ "id" => 1012, "name" => "Устюжна"],
            [ "id" => 1013, "name" => "Уфа"],
            [ "id" => 1014, "name" => "Ухта"],
            [ "id" => 1015, "name" => "Учалы"],
            [ "id" => 1016, "name" => "Уяр"],
            [ "id" => 1017, "name" => "Фатеж"],
            [ "id" => 1018, "name" => "Феодосия"],
            [ "id" => 1019, "name" => "Фокино"],
            [ "id" => 1020, "name" => "Фокино"],
            [ "id" => 1021, "name" => "Фролово"],
            [ "id" => 1022, "name" => "Фрязино"],
            [ "id" => 1023, "name" => "Фурманов"],
            [ "id" => 1024, "name" => "Хабаровск"],
            [ "id" => 1025, "name" => "Хадыженск"],
            [ "id" => 1026, "name" => "Ханты-Мансийск"],
            [ "id" => 1027, "name" => "Харабали"],
            [ "id" => 1028, "name" => "Харовск"],
            [ "id" => 1029, "name" => "Хасавюрт"],
            [ "id" => 1030, "name" => "Хвалынск"],
            [ "id" => 1031, "name" => "Хилок"],
            [ "id" => 1032, "name" => "Химки"],
            [ "id" => 1033, "name" => "Холм"],
            [ "id" => 1034, "name" => "Холмск"],
            [ "id" => 1035, "name" => "Хотьково"],
            [ "id" => 1036, "name" => "Цивильск"],
            [ "id" => 1037, "name" => "Цимлянск"],
            [ "id" => 1038, "name" => "Циолковский"],
            [ "id" => 1039, "name" => "Чадан"],
            [ "id" => 1040, "name" => "Чайковский"],
            [ "id" => 1041, "name" => "Чапаевск"],
            [ "id" => 1042, "name" => "Чаплыгин"],
            [ "id" => 1043, "name" => "Чебаркуль"],
            [ "id" => 1044, "name" => "Чебоксары"],
            [ "id" => 1045, "name" => "Чегем"],
            [ "id" => 1046, "name" => "Чекалин"],
            [ "id" => 1047, "name" => "Челябинск"],
            [ "id" => 1048, "name" => "Чердынь"],
            [ "id" => 1049, "name" => "Черемхово"],
            [ "id" => 1050, "name" => "Черепаново"],
            [ "id" => 1051, "name" => "Череповец"],
            [ "id" => 1052, "name" => "Черкесск"],
            [ "id" => 1053, "name" => "Чёрмоз"],
            [ "id" => 1054, "name" => "Черноголовка"],
            [ "id" => 1055, "name" => "Черногорск"],
            [ "id" => 1056, "name" => "Чернушка"],
            [ "id" => 1057, "name" => "Черняховск"],
            [ "id" => 1058, "name" => "Чехов"],
            [ "id" => 1059, "name" => "Чистополь"],
            [ "id" => 1060, "name" => "Чита"],
            [ "id" => 1061, "name" => "Чкаловск"],
            [ "id" => 1062, "name" => "Чудово"],
            [ "id" => 1063, "name" => "Чулым"],
            [ "id" => 1064, "name" => "Чусовой"],
            [ "id" => 1065, "name" => "Чухлома"],
            [ "id" => 1066, "name" => "Шагонар"],
            [ "id" => 1067, "name" => "Шадринск"],
            [ "id" => 1068, "name" => "Шали"],
            [ "id" => 1069, "name" => "Шарыпово"],
            [ "id" => 1070, "name" => "Шарья"],
            [ "id" => 1071, "name" => "Шатура"],
            [ "id" => 1072, "name" => "Шахты"],
            [ "id" => 1073, "name" => "Шахунья"],
            [ "id" => 1074, "name" => "Шацк"],
            [ "id" => 1075, "name" => "Шебекино"],
            [ "id" => 1076, "name" => "Шелехов"],
            [ "id" => 1077, "name" => "Шенкурск"],
            [ "id" => 1078, "name" => "Шилка"],
            [ "id" => 1079, "name" => "Шимановск"],
            [ "id" => 1080, "name" => "Шиханы"],
            [ "id" => 1081, "name" => "Шлиссельбург"],
            [ "id" => 1082, "name" => "Шумерля"],
            [ "id" => 1083, "name" => "Шумиха"],
            [ "id" => 1084, "name" => "Шуя"],
            [ "id" => 1085, "name" => "Щёкино"],
            [ "id" => 1086, "name" => "Щёлкино"],
            [ "id" => 1087, "name" => "Щёлково"],
            [ "id" => 1088, "name" => "Щигры"],
            [ "id" => 1089, "name" => "Щучье"],
            [ "id" => 1090, "name" => "Электрогорск"],
            [ "id" => 1091, "name" => "Электросталь"],
            [ "id" => 1092, "name" => "Электроугли"],
            [ "id" => 1093, "name" => "Элиста"],
            [ "id" => 1094, "name" => "Энгельс"],
            [ "id" => 1095, "name" => "Эртиль"],
            [ "id" => 1096, "name" => "Югорск"],
            [ "id" => 1097, "name" => "Южа"],
            [ "id" => 1098, "name" => "Южно-Сахалинск"],
            [ "id" => 1099, "name" => "Южно-Сухокумск"],
            [ "id" => 1100, "name" => "Южноуральск"],
            [ "id" => 1101, "name" => "Юрга"],
            [ "id" => 1102, "name" => "Юрьев-Польский"],
            [ "id" => 1103, "name" => "Юрьевец"],
            [ "id" => 1104, "name" => "Юрюзань"],
            [ "id" => 1105, "name" => "Юхнов"],
            [ "id" => 1106, "name" => "Ядрин"],
            [ "id" => 1107, "name" => "Якутск"],
            [ "id" => 1108, "name" => "Ялта"],
            [ "id" => 1109, "name" => "Ялуторовск"],
            [ "id" => 1110, "name" => "Янаул"],
            [ "id" => 1111, "name" => "Яранск"],
            [ "id" => 1112, "name" => "Яровое"],
            [ "id" => 1113, "name" => "Ярославль"],
            [ "id" => 1114, "name" => "Ярцево"],
            [ "id" => 1115, "name" => "Ясногорск"],
            [ "id" => 1116, "name" => "Ясный"],
            [ "id" => 1117, "name" => "Яхрома"],
        ];

        return response()->json($cities);
    }
}
