<?php
namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\PlanFile;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class GeneratePlanFileThumbnails extends Command
{
    protected $signature = 'planfiles:generate-thumbnails {--disk=s3}';
    protected $description = 'Generate or regenerate 500px JPG thumbnails (75% quality) for photo and PDF PlanFile entries';

    public function handle()
    {
        $diskName = $this->option('disk');
        $disk     = Storage::disk($diskName);

        // возьмём только фото (before/after) без thumbnail
        $files = PlanFile::whereIn('type', ['before', 'after', 'doc'])
            ->whereNull('thumbnail')
            ->get();

        if ($files->isEmpty()) {
            $this->info('Нет фотозаписей without thumbnail для обработки.');
            return 0;
        }

        foreach ($files as $planFile) {
            $this->info("Processing PlanFile ID: {$planFile->id}");
            try {
                $pathValue = $planFile->path;
                // Определяем URL к оригиналу
                $url = Str::startsWith($pathValue, ['http://','https://'])
                    ? $pathValue
                    : $disk->url($pathValue);

                // Скачиваем исходный файл во временный
                $tmpSource = tempnam(sys_get_temp_dir(), 'src_');
                file_put_contents($tmpSource, file_get_contents($url));

                // Создаём Imagick-объект
                $imagick = new \Imagick();
                $ext = strtolower(pathinfo($pathValue, PATHINFO_EXTENSION));

                if ($ext === 'pdf') {
                    // PDF → только первая страница с высоким разрешением
                    $imagick->setResolution(150, 150);
                    $imagick->readImage("{$tmpSource}[0]");
                } else {
                    // Изображение (jpg, png) → читаем всё
                    $imagick->readImage($tmpSource);
                }

                // Ресайз: большая сторона = 500px
                $w = $imagick->getImageWidth();
                $h = $imagick->getImageHeight();
                if ($w >= $h) {
                    $imagick->resizeImage(500, 0, \Imagick::FILTER_LANCZOS, 1);
                } else {
                    $imagick->resizeImage(0, 500, \Imagick::FILTER_LANCZOS, 1);
                }

                // Перекодируем в JPEG 75%
                $imagick->setImageFormat('jpeg');
                $imagick->setImageCompression(\Imagick::COMPRESSION_JPEG);
                $imagick->setImageCompressionQuality(75);

                // Сохраняем превью в tmp и загружаем на S3
                $filename = pathinfo($pathValue, PATHINFO_FILENAME);
                $thumbFilename = "thumb_{$filename}.jpg";
                $tmpThumb = sys_get_temp_dir() . DIRECTORY_SEPARATOR . $thumbFilename;
                $imagick->writeImage($tmpThumb);
                $imagick->clear();
                $imagick->destroy();
                @unlink($tmpSource);

                $thumbPath = "plan_files/{$thumbFilename}";
                $disk->put($thumbPath, file_get_contents($tmpThumb), 'public');
                @unlink($tmpThumb);

                // Сохраняем путь в БД
                $planFile->thumbnail = $thumbPath;
                $planFile->save();

                $this->info("Thumbnail created for PlanFile ID: {$planFile->id}");
            } catch (\Exception $e) {
                $this->error("Error processing ID {$planFile->id}: {$e->getMessage()}");
            }
        }

        $this->info('Генерация превьюшек завершена.');
        return 0;
    }
}
