<?php
namespace App\Exports;

use App\Services\TrafficReportCalculator;
use Maatwebsite\Excel\Concerns\FromArray;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Carbon\Carbon;
use App\Models\PointPlan;

class TrafficReportPeriodExport implements FromArray, WithHeadings
{
    protected $period;

    public function __construct(array $period)
    {
        $this->period = $period;
    }

    public function array(): array
    {
        // 1) вычисляем дату-диапазон
        $year = $this->period['year'];
        switch ($this->period['periodType']) {
            case 'month':
                $mth = (int) $this->period['month'];
                $from = Carbon::create($year, $mth, 1)->startOfDay();
                $to   = Carbon::create($year, $mth, 1)->endOfMonth();
                break;
            case 'quarter':
                $q = (int)$this->period['quarter'];
                $startM = ($q - 1) * 3 + 1;
                $from = Carbon::create($year, $startM, 1)->startOfDay();
                $to   = Carbon::create($year, $startM + 2, 1)->endOfMonth();
                break;
            case 'half':
                $h = (int)$this->period['half'];
                $startM = ($h - 1) * 6 + 1;
                $from = Carbon::create($year, $startM, 1)->startOfDay();
                $to   = Carbon::create($year, $startM + 5, 1)->endOfMonth();
                break;
            default:
                $from = Carbon::create($year, 1, 1)->startOfDay();
                $to   = Carbon::create($year, 12, 31)->endOfDay();
        }

        // 2) выбираем все планы, попадающие в период
        $plans = PointPlan::whereBetween('day', [$from->toDateString(), $to->toDateString()])
            ->with('measurements')
            ->get();

        // 3) подготавливаем шапку
        $rows = [];
        $rows[] = [
            'Категория',
            'Ni (шт за 4 ч)',
            'N₅₀ (шт/ч)',
            'Nсс (шт/сут)',
            'Легков. экв. (шт/сут)',
        ];

        // для итогов по всем строкам
        $totals = [
            'Ni'    => 0,
            'N50'   => 0,
            'Ncs'   => 0,
            'toCar' => 0,
        ];

        foreach ($plans as $plan) {
            // запускаем калькулятор
            $calc = (new TrafficReportCalculator($plan))->calculate();
            // сколько часов в замере
            $m = $calc['meta']['m'];

            // сгруппируем сырые измерения по категории, чтобы взять Ni именно для неё
            $byCat = $plan->measurements->groupBy('category_key');

            foreach ($calc['detail'] as $categoryKey => $data) {
                // 1) Ni — суммарное за 4 ч по этой категории
                $Ni = $byCat[$categoryKey]->sum('count');

                // 2) N50 = Ni / m
                $N50 = $m > 0 ? $Ni / $m : 0;

                // 3) Ncs и to_car приходят из калькулятора
                $Ncs   = $data['Ncs']    ?? 0;
                $toCar = $data['to_car'] ?? 0;

                // добавляем строку
                $rows[] = [
                    $categoryKey,
                    $Ni,
                    round($N50,   2),
                    round($Ncs,   2),
                    round($toCar, 2),
                ];

                // накапливаем итоги
                $totals['Ni']    += $Ni;
                $totals['N50']   += $N50;
                $totals['Ncs']   += $Ncs;
                $totals['toCar'] += $toCar;
            }
        }

        // итоговая строка
        $rows[] = [
            'Итого',
            $totals['Ni'],
            round($totals['N50'],   2),
            round($totals['Ncs'],   2),
            round($totals['toCar'], 2),
        ];

        return $rows;
    }

    public function headings(): array
    {
        // мы вставили шапку вручную, здесь можно вернуть пустой массив
        return [];
    }
}
