<?php

namespace App\Http\Controllers;

use Spatie\Activitylog\Models\Activity;
use Illuminate\Http\Request;

class ActivityLogController extends Controller
{
    public function index(Request $request)
    {
        $perPage = $request->query('per_page', 100);

        // базовый запрос
        $query = Activity::with('causer')
            ->orderBy('created_at', 'desc');

        // --- НАЧАЛО ДОБАВОК: фильтры ---

        // 1) По объекту (log_name)
        if ($obj = $request->query('object')) {
            $query->where('log_name', $obj);
        }

        // 2) По действию (description)
        if ($act = $request->query('action')) {
            $query->where('description', $act);
        }

        // 3) По автору (causer_id)
        if ($uid = $request->query('author_id')) {
            $query->where('causer_id', $uid);
        }

        // 4) По дате: от
        if ($from = $request->query('date_from')) {
            $query->where('created_at', '>=', $from);
        }
        //    по
        if ($to = $request->query('date_to')) {
            $query->where('created_at', '<=', $to);
        }

        // получаем «ленивую» пагинацию
        $paginator = $query->paginate($perPage);

        // маппинг модели в нужный вам формат
        $data = $paginator->getCollection()->map(function(Activity $act) {
            return [
                'id'          => $act->id,
                'log_name'    => $act->log_name,
                'description' => $act->description,
                'subject'     => [
                    'type' => class_basename($act->subject_type),
                    'id'   => $act->subject_id,
                ],
                'old'         => $act->properties['old'] ?? [],
                'new'         => $act->properties['attributes'] ?? [],
                'causer'      => $act->causer
                    ? ['id' => $act->causer->id, 'name' => $act->causer->full_name]
                    : null,
                'created_at'  => $act->created_at->toIso8601String(),
            ];
        });

        // возвращаем текущую «страницу» и мета-данные
        return response()->json([
            'logs' => $data,
            'meta' => [
                'current_page' => $paginator->currentPage(),
                'last_page'    => $paginator->lastPage(),
                'per_page'     => $paginator->perPage(),
                'total'        => $paginator->total(),
            ],
        ]);
    }

    public function objects()
    {
        $objects = Activity::select('log_name')
            ->distinct()
            ->orderBy('log_name')
            ->pluck('log_name');

        return response()->json([
            'objects' => $objects,
        ]);
    }

}
