<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\Auth\LoginRequest;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;
use App\Models\Role\Role;
use App\Models\Role\Permission;

use function PHPSTORM_META\map;

class AuthenticatedSessionController extends Controller
{
    
    /**
     * Запрос на авторизацию по логину и паролю
     *
     * @param LoginRequest $request
     * @return JsonResponse
     * @throws ValidationException
     */

     public function signIn(LoginRequest $request): JsonResponse
     {
         if (!Auth::guard('web')->attempt($request->only('login', 'password'))) {
             throw ValidationException::withMessages([
                 'login' => ['Неверный логин или пароль.'],
             ]);
         }
     
         /** @var User $user */
         $user = Auth::user();
         // Получаем объект Role, связанный с пользователем
         $userRole = $user->role; 
         // Если роль есть, получаем коллекцию permissions (через связь многие-ко-многим)
         $rolePermissions = $userRole->permissions()->get()->map(function ($permission) {
            return $permission->name;
        });

         // Создаём Sanctum-токен
         $token = $user->createToken('auth_token')->plainTextToken;
     
         return response()->json([
             'data' => [
                
                'accessToken' => $token,
                'tokenType'   => 'Bearer',
                'user'        => [
                    'id' => $user->id,
                    'first_name' => $user->first_name,
                    'last_name' => $user->last_name,
                    'middle_name' => $user->middle_name,
                    'login' => $user->login,
                    'position' => $user->position,
                    'birthday' => $user->birthday,
                    'private_key' => $user->private_key,
                    'public_key' => $user->public_key,
                    'refresh_token' => $user->refresh_token,
                    'token' => $user->token,
                    'role' => $userRole,
                    'rolePermissions' => $rolePermissions,
                ],
             ]
         ]);
     }

    /**
     * Выход пользователя (не обязателен, если делаешь logout через routes/api.php)
     */
    public function destroy(Request $request): JsonResponse
{
    if ($request->user()) {
        $request->user()->currentAccessToken()->delete();
    }

    return response()->json([
        'message' => 'Выход произведен успешно',
    ]);
}
}
