<?php

namespace App\Http\Controllers;

use App\Models\Counter;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;

class CounterController extends Controller
{
    public function index()
    {
        $user = Auth::user();

        // Если роль пользователя — Менеджер подгруппы, возвращаем только "своих" учётчиков
        if ($user->role && $user->role->name === 'Менеджер подгруппы') {
            return Counter::where('subgroup_manager_id', $user->id)
                ->with('subgroupManager:id,first_name,middle_name,last_name')
                ->get();
        }

        // Иначе (администраторы, разработчики и т.д.) — полный список
        return Counter::with('subgroupManager:id,first_name,middle_name,last_name')->get();
    }
    public function store(Request $r)
    {
        $data = $r->validate([
            'first_name'           => ['required', 'string', 'max:255'],
            'middle_name'          => ['nullable', 'string', 'max:255'],
            'last_name'            => ['required', 'string', 'max:255'],
            'subgroup_manager_id'  => [
                'required',
                'integer',
                Rule::exists('users', 'id')->where(function ($query) {
                    // проверяем, что этот пользователь действительно Менеджер подгруппы
                    $query->where('role_id', function ($q) {
                        // подставьте здесь ID роли «Менеджер подгруппы»,
                        // либо сравните по имени роли в таблице roles
                        // например: id=5  (в вашем примере)
                        $q->select('id')
                            ->from('roles')
                            ->where('name', 'Менеджер подгруппы');
                    });
                }),
            ],
        ]);

        // Создаём нового учётчика
        $counter = Counter::create($data);

        return response()->json($counter, 201);
    }

    public function update(Request $r, Counter $counter)
    {
        $data = $r->validate([
            'first_name'           => ['required', 'string', 'max:255'],
            'middle_name'          => ['nullable', 'string', 'max:255'],
            'last_name'            => ['required', 'string', 'max:255'],
            'subgroup_manager_id'  => [
                'required',
                'integer',
                Rule::exists('users', 'id')->where(function ($query) {
                    $query->where('role_id', function ($q) {
                        $q->select('id')
                            ->from('roles')
                            ->where('name', 'Менеджер подгруппы');
                    });
                }),
            ],
        ]);

        $counter->update($data);
        return response()->json($counter);
    }
    public function destroy(Counter $counter)
    {
        // 1) Получаем все PointPlan, где задействован этот учётчик,
        //    вместе с привязанными Point (чтобы узнать record_number и name).
        $plans = $counter->pointPlans()->with('point:id,record_number,name')->get();

        // 2) Если есть связанные планы, собираем список уникальных точек:
        if ($plans->isNotEmpty()) {
            // Сбор уникальных точек (по id), чтобы не дублировать, если учётчик
            // задействован на одной точке несколько раз
            $uniquePoints = $plans
                ->map(fn($plan) => $plan->point)
                ->unique('id')
                ->map(fn($point) => "{$point->record_number} («{$point->name}»)")
                ->values();

            $listStr = $uniquePoints->join(', ');

            return response()->json([
                'message' => "Нельзя удалить учётчика: он назначен на следующие точки: $listStr."
            ], 400);
        }

        // 3) Если связей нет — удаляем как обычно
        $counter->delete();
        return response()->noContent();
    }
}
