<?php

namespace App\Http\Controllers;

use App\Models\PlanFile;
use App\Models\PointPlan;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class PlanFileController extends Controller
{
    /**
     * Загрузить файл (JPEG/PNG/PDF), сохранить на S3,
     * при PDF конвертировать первую страницу в JPG превью с максимальной стороной 500px и качеством 75%.
     * При загрузке изображений JPEG/PNG также генерировать превью 500px.
     */
    public function upload(Request $request)
    {
        $request->validate([
            'point_plan_id' => 'required|integer|exists:point_plans,id',
            'type'          => 'required|in:before,after,doc',
            'file'          => 'required|file|mimes:jpg,jpeg,png,pdf|max:102400',
        ]);

        $file   = $request->file('file');
        $planId = $request->input('point_plan_id');
        $type   = $request->input('type');
        $disk   = Storage::disk('s3');
        $folder = 'plan_files';

        // Получаем объект плана и заявку
        $plan = PointPlan::findOrFail($planId);
        $request = $plan->request;

        if ($request && $request->is_locked) {
            // Проверяем роль пользователя (региональный менеджер или разработчик)
            $user = Auth::user();
            if (!$user->hasRole('Региональный менеджер') && !$user->hasRole('Разработчик')) {
                return response()->json([
                    'message' => 'Заявка подписана. Только региональный менеджер может загружать файлы.'
                ], 403);
            }
        }

        // 1) Сохраняем оригинал на S3 и получаем полный URL
        $key  = $file->store($folder, 's3');
        $url  = $disk->url($key);

        // 2) Генерируем превью через Imagick
        $imagick = new \Imagick();
        $ext     = strtolower($file->getClientOriginalExtension());

        if ($ext === 'pdf') {
            $imagick->setResolution(150,150);
            $imagick->readImage($file->getRealPath() . '[0]');
        } else {
            $imagick->readImage($file->getRealPath());
        }

        // ресайз до 500px по большей стороне
        $w = $imagick->getImageWidth();
        $h = $imagick->getImageHeight();
        if ($w >= $h) {
            $imagick->resizeImage(500, 0, \Imagick::FILTER_LANCZOS, 1);
        } else {
            $imagick->resizeImage(0, 500, \Imagick::FILTER_LANCZOS, 1);
        }

        // конверт в JPEG с качеством 75%
        $imagick->setImageFormat('jpeg');
        $imagick->setImageCompression(\Imagick::COMPRESSION_JPEG);
        $imagick->setImageCompressionQuality(75);

        // сохраняем превью в tmp и заливаем на S3
        $thumbName = 'thumb_' . pathinfo($key, PATHINFO_FILENAME) . '.jpg';
        $thumbKey  = "$folder/$thumbName";
        $tmp       = sys_get_temp_dir() . "/$thumbName";
        $imagick->writeImage($tmp);
        $disk->put($thumbKey, file_get_contents($tmp), 'public');
        @unlink($tmp);
        $imagick->clear();
        $imagick->destroy();

        $thumbUrl = $thumbKey;

        // 3) Сохраняем в БД полный URL и URL превью
        $planFile = PlanFile::create([
            'point_plan_id' => $planId,
            'type'          => $type,
            'path'          => $url,        // полный URL
            'thumbnail'     => $thumbUrl,   // полный URL превью
        ]);

        // 4) Обновляем статус PointPlan
        $plan = PointPlan::find($planId);
        if (collect(['before','after'])->every(fn($t) => $plan->files()->where('type',$t)->exists())
            && !$plan->isWorkedOut
        ) {
            $plan->update([
                'isWorkedOut' => true,
                'signature'   => Auth::user()->name,
            ]);
        }

        return response()->json([
            'id'           => $planFile->id,
            'url'          => $url,
            'thumbnailUrl' => $thumbUrl,
        ], 201);
    }



    /**
     * Выдать список файлов для point_plan (id, type, url, thumbnailUrl)
     */
    public function list(int $planId)
    {
        $disk = Storage::disk('s3');

        $files = PlanFile::where('point_plan_id', $planId)->get()->map(fn($f) => [
            'id'           => $f->id,
            'type'         => $f->type,
            'url'          => $f->path,
            'thumbnailUrl' => $disk->url($f->thumbnail),
        ]);

        return response()->json(['files' => $files]);
    }

    /**
     * Удалить файл и его превью (если есть)
     */
    public function destroy(int $id)
    {
        $file = PlanFile::findOrFail($id);

        $plan = $file->plan;
        $requestSet = $plan->request;

        // Проверка: если заявка подписана — только региональный менеджер или разработчик может удалять
        if ($requestSet && $requestSet->is_locked) {
            $user = Auth::user();
            if (
                !$user->hasRole('Региональный менеджер')
                && !$user->hasRole('Разработчик')
            ) {
                return response()->json([
                    'message' => 'Заявка подписана. Только региональный менеджер может удалять файлы.'
                ], 403);
            }
        }

        $disk = Storage::disk('s3');

        // Удаляем оригинал
        $disk->delete($file->path);
        // Удаляем превью
        if ($file->thumbnail) {
            $disk->delete($file->thumbnail);
        }

        $file->delete();
        return response()->json(['message' => 'Deleted']);
    }
}
