<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreMeasurementRecordsRequest;
use App\Models\MeasurementRecord;
use App\Models\PointPlan;
use App\Models\RequestSet;
use App\Models\VehicleCategory;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class PlanStatsController extends Controller

{
    // список категорий ТС
    public function categories()
    {
        $categories = VehicleCategory::orderBy('sort_order')->get();
        return response()->json($categories);
    }

    // получить данные для заполнения модалки
    public function show(int $planId): JsonResponse
    {
        // 1) Берём план, чтобы получить его direction
        $plan = PointPlan::findOrFail($planId);

        // 2) Группируем записи по категории и собираем плоский мап час→количество
        $records = MeasurementRecord::where('point_plan_id', $planId)
            ->get()
            ->groupBy('category_key')
            ->map(function ($group) {
                // вернёт коллекцию вида ['08:00' => 11, '09:00' =>  2, …]
                return $group->pluck('count', 'hour');
            });

        // 3) Возвращаем JSON с отдельным полем direction и мапом records
        return response()->json([
            'direction' => $plan->direction,
            'weather' => $plan->weather,
            'lanes_count' => $plan->lanes_count,
            'records' => $records,
        ]);
    }


    // сохранить
    public function store(StoreMeasurementRecordsRequest $req, int $planId)
    {
        // 1) Обновляем направление у самого плана
        $plan = PointPlan::findOrFail($planId);

        $request = $plan->request ?? RequestSet::find($plan->request_id);
        if (!$request) {
            return response()->json(['message' => 'Заявка не найдена'], 404);
        }
        if ($request->is_locked) {
            // Проверяем роль пользователя (например, admin, regional_manager)
            $user = auth()->user();
            if (!$user->hasRole('Региональный менеджер') && !$user->hasRole('Разработчик')) {
                return response()->json(['message' => 'Заявка подписана. Вносить изменения может только региональный менеджер.'], 403);
            }
        }

        $plan->direction = $req->input('direction');
        $plan->weather = $req->input('weather');
        $plan->lanes_count = $req->input('lanes_count');

        $plan->is_manual = $req->boolean('is_manual');
        $plan->save();

        // 2) Стираем старые записи
        MeasurementRecord::where('point_plan_id', $planId)->delete();

        // 3) Создаём новые, без поля direction
        foreach ($req->input('records') as $r) {
            MeasurementRecord::create([
                'point_plan_id' => $planId,
                'category_key' => $r['category_key'],
                'hour' => $r['hour'],
                'count' => $r['count'],
            ]);
        }

        return response()->json(['message' => 'OK']);
    }

    // получить замеры за тот же месяц прошлого года для данной точки
    public function getPreviousStats(Request $request): JsonResponse
    {
        $pointId = $request->input('point_id');
        $month   = (int)$request->input('month');
        $year    = (int)$request->input('year');

        // 1) сначала смотрим на тот же месяц прошлого года, но только ручные записи
        $planIds = PointPlan::where('point_id', $pointId)
            ->whereYear('day', $year)
            ->whereMonth('day', $month)
            ->where('is_manual', true)
            ->pluck('id');

        if ($planIds->isNotEmpty()) {
            return response()->json($this->formatStats($planIds));
        }

        // 2) если нет — ищем в прошлом месяце (от текущей даты), только ручные
        $prev = \Carbon\Carbon::now()->subMonth();
        $planIds = PointPlan::where('point_id', $pointId)
            ->whereYear('day', $prev->year)
            ->whereMonth('day', $prev->month)
            ->where('is_manual', true)
            ->pluck('id');

        if ($planIds->isNotEmpty()) {
            return response()->json($this->formatStats($planIds));
        }

        // 3) если и тут пусто — берём последний по дате ручной план
        $lastPlan = PointPlan::where('point_id', $pointId)
            ->where('is_manual', true)
            ->orderBy('day', 'desc')
            ->first();

        if (!$lastPlan) {
            return response()->json([]);
        }

        $planIds = PointPlan::where('point_id', $pointId)
            ->whereYear('day', $lastPlan->day->year)
            ->whereMonth('day', $lastPlan->day->month)
            ->where('is_manual', true)
            ->pluck('id');

        return response()->json($this->formatStats($planIds));
    }



    /**
     * Вспомогательный метод для извлечения и форматирования записей.
     */
    private function formatStats($planIds): array
    {
        $records = MeasurementRecord::whereIn('point_plan_id', $planIds)->get();

        $data = [];
        foreach ($records->groupBy('category_key') as $catKey => $group) {
            $hours = [];
            foreach ($group as $rec) {
                $hours[$rec->hour] = $rec->count;
            }
            $data[] = [
                'category_key' => $catKey,
                'hours'        => $hours,
            ];
        }
        return $data;
    }



}
