<?php

namespace App\Http\Controllers;

use App\Http\Resources\PointResource;
use App\Models\Point;
use App\Models\RequestSet;
use Illuminate\Http\Request;

class PointController extends Controller
{
    // GET /api/v1/points
    public function index()
    {
        $points = Point::with(['district.region'])->paginate();
        return PointResource::collection($points);
    }

    // POST /api/v1/points
    public function store(Request $request)
    {
        $data = $request->validate([
            'name'            => 'required|string',
            'latitude'        => 'required|numeric',
            'longitude'       => 'required|numeric',
            'district_id'     => 'required|exists:districts,id',
            'segment_from'    => 'nullable|numeric',
            'segment_to'      => 'nullable|numeric',
            // Новые поля:
            'length'          => 'nullable|numeric',
            'accounting_point'=> 'nullable|string',
            'file'            => 'nullable|file|mimes:json,geojson',
        ]);

        if ($request->hasFile('file')) {
            $geojsonContent = $request->file('file')->get();
            $data['geojson'] = json_decode($geojsonContent, true);
        }

        $point = Point::create($data);
        $point->load('district.region');

        return new PointResource($point);
    }

    // GET /api/v1/points/{point}
    public function show(Point $point)
    {
        $point->load('district.region');
        return new PointResource($point);
    }

    // PATCH /api/v1/points/{point}
    public function update(Request $request, Point $point)
    {
        $data = $request->validate([
            'name'            => 'required|string',
            'latitude'        => 'required|numeric',
            'longitude'       => 'required|numeric',
            'district_id'     => 'required|exists:districts,id',
            'segment_from'    => 'nullable|numeric',
            'segment_to'      => 'nullable|numeric',
            // Новые поля:
            'length'          => 'nullable|numeric',
            'accounting_point'=> 'nullable|string',
            'file'            => 'nullable|file|mimes:json,geojson',
        ]);

        $lockedRequest = $point->requests()->where('is_locked', true)->first();
        if ($lockedRequest) {
            $user = auth()->user();
            if (!$user->hasRole('Региональный менеджер') && !$user->hasRole('Разработчик')) {
                return response()->json([
                    'message' => 'Точка привязана к подписанной заявке. Вносить изменения может только региональный менеджер.'
                ], 403);
            }
        }

        if ($request->hasFile('file')) {
            $geojsonContent = $request->file('file')->get();
            $data['geojson'] = json_decode($geojsonContent, true);
        }

        $point->update($data);
        $point->load('district.region');

        return new PointResource($point);
    }

    // DELETE /api/v1/points/{point}
    public function destroy(Point $point)
    {
        $lockedRequest = $point->requests()->where('is_locked', true)->first();
        if ($lockedRequest) {
            $user = auth()->user();
            if (!$user->hasRole('Региональный менеджер') && !$user->hasRole('Разработчик')) {
                return response()->json([
                    'message' => 'Точка привязана к подписанной заявке. Вносить изменения может только региональный менеджер.'
                ], 403);
            }
        }

        $point->delete();
        return response()->noContent();
    }

    public function names()
    {
        // отдаём только id, name и record_number
        $list = Point::select('id','name','record_number')
            ->orderBy('record_number')
            ->get();

        return response()->json(['points' => $list]);
    }
}
