<?php

namespace App\Http\Controllers;

use App\Models\PointPlan;
use Carbon\Carbon;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class PointPlanController extends Controller
{
    public function index(Request $r): JsonResponse
    {
        $validated = $r->validate([
            'point_id'   => ['required','exists:points,id'],
            'request_id' => ['required','exists:requests,id'],
            'schedule_id'=> ['required','exists:schedule_point_plans,id'],
        ]);

        $plans = PointPlan::where([
            'point_id'     => $validated['point_id'],
            'request_id'   => $validated['request_id'],
            'schedule_id'  => $validated['schedule_id'],
        ])->get();

        return response()->json(['plans' => $plans]);
    }

    public function summary(Request $r): JsonResponse
    {
        $validated = $r->validate([
            'point_id'    => ['required', 'exists:points,id'],
            'request_id'  => ['required', 'exists:requests,id'],
            'schedule_id' => ['required', 'exists:schedule_point_plans,id'],
        ]);

        // Базовый запрос с фильтрами
        $baseQuery = PointPlan::where([
            'point_id'    => $validated['point_id'],
            'request_id'  => $validated['request_id'],
            'schedule_id' => $validated['schedule_id'],
        ]);

        // Общее число задач
        $total = $baseQuery->count();

        // Выполнено
        $completed = (clone $baseQuery)
            ->where('isWorkedOut', true)
            ->count();

        // С замечаниями
        $remarks = (clone $baseQuery)
            ->whereNotNull('comment')
            ->count();

        // Просрочено: дата меньше сегодня и ещё не обработано
        $today = Carbon::today()->toDateString();
        $overdue = (clone $baseQuery)
            ->whereDate('day', '<', $today)
            ->where('isWorkedOut', false)
            ->count();

        // Осталось (не выполнено)
        $remaining = $total - $completed;

        // Дедлайн — максимальная дата из выборки
        $maxDay = (clone $baseQuery)->max('day');
        $deadline = $maxDay
            ? Carbon::parse($maxDay)->toDateString()
            : null;

        return response()->json([
            'planned'   => $total,
            'completed' => $completed,
            'remarks'   => $remarks,
            'overdue'   => $overdue,
            'remaining' => $remaining,
            'deadline'  => $deadline,
        ]);
    }

}
