<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Spatie\Activitylog\LogOptions;
use Spatie\Activitylog\Traits\LogsActivity;

class PointPlan extends Model
{
    use LogsActivity;

    protected $table = 'point_plans';

    protected $fillable = [
        'day',
        'timeStart',
        'timeEnd',
        'point_id',
        'schedule_id',
        'subgroup_manager_id',
        'counter_id',
        'request_id',
        'isWorkedOut',
        'is_active',
        'comment',
        'signature',
        'signature_public_key',
        'hash',
        'direction',
        'weather',
        'lanes_count',
        'is_manual',
    ];
    protected $casts = [
        'day' => 'date',
        'timeStart' => 'datetime:H:i',
        'timeEnd' => 'datetime:H:i',
        'counter_id' => 'integer',
        'direction' => 'integer',
        'weather' => 'string',
        'is_manual' => 'boolean',
        'lanes_count' => 'integer',
    ];

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->useLogName('Работы')
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs()
            ->logOnly([
                'day',
                'timeStart',
                'timeEnd',
                'point_id',
                'schedule_id',
                'subgroup_manager_id',
                'counter_id',
                'request_id',
                'isWorkedOut',
                'is_active',
                'comment',
                'signature',
                'signature_public_key',
                'hash',
                'direction',
                'weather',
                'lanes_count',
                'is_manual',
            ]);
    }


    public function point()
    {
        return $this->belongsTo(Point::class, 'point_id');
    }

    public function schedule()
    {
        return $this->belongsTo(SchedulePlan::class, 'schedule_id');
    }

    public function subgroupManager(): BelongsTo
    {
        return $this->belongsTo(User::class, 'subgroup_manager_id');
    }

    /**
     * Файлы, загруженные к этой точке
     */
    public function files()
    {
        return $this->hasMany(PlanFile::class, 'point_plan_id');
    }

    public function counter()
    {
        return $this->belongsTo(Counter::class, 'counter_id');
    }

    public function request()
    {
        return $this->belongsTo(RequestSet::class, 'request_id');
    }

    public function measurements()
    {
        return $this->hasMany(MeasurementRecord::class, 'point_plan_id');
    }

    public function scopeForGroupManager($q, int $mgrId): void
    {
        $q->whereHas('point.district', fn($q) => $q
            ->where('group_manager_id', $mgrId)
        );
    }

    public function scopeForSubgroupManager($q, int $mgrId): void
    {
        $q->where(function ($q) use ($mgrId) {
            $q->where('subgroup_manager_id', $mgrId)          // сам план
            ->orWhereHas('point.district', fn($q) => $q     // или район
            ->where('subgroup_manager_id', $mgrId)
            );
        });
    }

    public function scopeForDistrict($q, int $districtId): void
    {
        $q->whereHas('point', fn($q) => $q->where('district_id', $districtId));
    }

}
