<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use App\Models\Role\Role;
use App\Models\Role\Permission;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Laravel\Sanctum\HasApiTokens;
use Spatie\Activitylog\LogOptions;
use Spatie\Activitylog\Traits\LogsActivity;


class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasApiTokens, HasFactory, Notifiable;
    use LogsActivity;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'id',
        'first_name',
        'middle_name',
        'last_name',
        'is_active',
        'position',
        'login',
        'birthday',
        'role_id',
        'password',
        'token',
        'refresh_token',
        'public_key',
        'private_key',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
        'public_key',
        'private_key',
    ];

    protected $appends = ['full_name'];   // будет виден при toArray()


    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->useLogName('Пользователи')
            ->logOnlyDirty()                     // только изменённые поля
            ->dontSubmitEmptyLogs()              // не писать «пустые» логи
            ->logOnly([
                'first_name',
                'middle_name',
                'last_name',
                'is_active',
                'position',
                'role_id',
                'login',
                'birthday',
                'password',

            ])
            ->setDescriptionForEvent(fn(string $eventName) => "Пользователь был {$eventName}");
    }

    /**
     * Проверяет, что у пользователя есть роль по имени или по ID.
     *
     * @param  string|int  $roleNameOrId
     * @return bool
     */
    public function hasRole($roleNameOrId): bool
    {
        // Если передали число — сравниваем по role_id
        if (is_int($roleNameOrId) || ctype_digit((string)$roleNameOrId)) {
            return $this->role_id === (int)$roleNameOrId;
        }

        // Иначе сравниваем по имени роли через связь
        return optional($this->role)->name === $roleNameOrId;
    }

    /**
     * Проверяет, что у пользователя есть указанное право.
     *
     * @param  string  $permissionName
     * @return bool
     */
    public function hasPermission(string $permissionName): bool
    {
        // Загружаем связи роли → права
        // Если у вас eager-load, можно не перегружать
        $role = $this->role()->with('permissions')->first();

        if (! $role) {
            return false;
        }

        return $role->permissions->contains('name', $permissionName);
    }

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'password' => 'hashed',
        ];
    }

    public function managedRegions(){
        return $this->hasMany(Region::class, 'sen_manager_id');
    }

    public function managedPointPlan()
    {
        return $this->hasMany(PointPlan::class, 'subgroup_manager_id');
    }

    public function managedScheduls()
    {
        return $this->hasMany(SchedulePlan::class, 'sen_manager_id');
    }

    public function fullName(): string
    {
        return $this->middle_name . ' ' . $this->first_name . ' ' . $this->last_name;
    }
    public function role(): BelongsTo
    {
        return $this->belongsTo(Role::class, 'role_id');
    }



    public function userNotifications(){
        return $this->hasMany(Notification::class, 'user_id');
    }
    public function createdLogs(){
        return $this->hasMany(Log::class, 'created_by');
    }
    public function targetLogs(){
        return $this->hasMany(Log::class, 'model_id');
    }

    public function scopeGroupManagers($q)
    {
        return $q->whereHas('role', fn($r) =>
        $r->where('name', 'Менеджер группы')
            ->where('is_active', true)
        );
    }

    public function scopeSubgroupManagers($q)
    {
        return $q->whereHas('role', fn($r) =>
        $r->where('name', 'Менеджер подгруппы')
            ->where('is_active', true)
        );
    }

    public function scopeActive($q)
    {
        $q->where('is_active', true);
    }

    public function managedDistricts()   // менеджеры групп
    {
        return $this->hasMany(District::class, 'group_manager_id');
    }

    public function managedSubgroupDistricts()   // если понадобится отдельно
    {
        return $this->hasMany(District::class, 'subgroup_manager_id');
    }



    public function getFullNameAttribute(): string
    {
        return trim("{$this->last_name} {$this->first_name} {$this->middle_name}");
    }
}



