<?php
namespace App\Services;

use App\Models\PointPlan;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class TrafficReportCalculator
{
    protected $plan;
    protected $coeff;

    public function __construct(PointPlan $plan)
    {
        $this->plan  = $plan;
        $this->coeff = config('traffic');
    }

    public function calculate()
    {
        // 0) все измерения
        $measurements = $this->plan->measurements()->get();
        if ($measurements->isEmpty()) {
            return [
                'detail'     => [],
                'totalNcs'   => 0,
                'totalCarEq' => 0,
                'K'          => ['K_M'=>0,'K_D'=>0,'K_H'=>0],
                'meta'       => ['m'=>0,'day'=>null,'month'=>null,'hourStart'=>null,'peak50'=>0],
            ];
        }

        // 1) берем минимальный и максимальный час из самих замеров
        $times = $measurements
            ->pluck('hour')               // допустим, '08:00', '12:00' и т.п.
            ->sort()                      // упорядочим
            ->map(fn($h) => Carbon::parse(
            // соединяем дату замера и само время
                $this->plan->day->format('Y-m-d') . ' ' . $h
            ));

        // Количество часов измерений
        $hours = 4;

        // 2) первый замер + день/месяц/час
        $firstRecord = $measurements->first();
        $dt          = Carbon::parse($this->plan->day->format('Y-m-d') . ' ' . $firstRecord->hour);
        $dayOfWeek = strtolower($dt->format('D'));
        $month     = $dt->month;
        $hourStart = $dt->hour;

        // 3) тип точки
        $type = $this->plan->point->accounting_flag === 1 ? 'podhod' : 'peregon';

        // 4) коэффициенты
        $K_M = $this->coeff['K_month'][$type][$month]       ?? 1;
        $K_D = $this->coeff['K_weekday'][$type][$dayOfWeek] ?? 1;
        $K_H = $this->coeff['K_hour'][$type][$hourStart][$hours] ?? 1;
        $KtoC = $this->coeff['K_to_car'];

        // 5) группировка
        $records= $measurements->groupBy('category_key');
        $detail = [];
        $totalNcs = $totalCarEq = 0;
        $allHourly = [];

        // после загрузки коэффициентов

        Log::debug('Traffic coef', [

            'month' => $month,
            'dayOfWeek' => $dayOfWeek,
            'hourStart' => $hourStart,
            'K_M' => $K_M,
            'K_D' => $K_D,
            'K_H' => $K_H,
        ]);

        foreach ($records as $cat => $group) {
            $Ni = $group->sum('count');                // за m часов
            $Ncs_i = (int) round($Ni * $K_H * $K_D * $K_M);    // Ncc
            $coef  = $KtoC[$cat] ?? 1;
            $toCar = (int) round($Ncs_i * ($KtoC[$cat] ?? 1));

            // Подробный лог
            Log::debug("Traffic calc for category {$cat}", [
                'Ni'     => $Ni,
                'K_M'    => $K_M,
                'K_D'    => $K_D,
                'K_H'    => $K_H,
                'Ncs_i'  => $Ncs_i,
                'K_toCar'=> $coef,
                'toCar'  => $toCar,
            ]);

            $detail[$cat] = ['Ncs'=>$Ncs_i, 'to_car'=>$toCar];
            $totalNcs   += $Ncs_i;
            $totalCarEq += $toCar;

            // для N50
            $allHourly = array_merge($allHourly, $group->pluck('count')->toArray());
        }

        $peak50 = (int) round($totalCarEq / 24);
        return [
            'detail'     => $detail,
            'totalNcs'   => $totalNcs,
            'totalCarEq' => $totalCarEq,
            'K'          => compact('K_M','K_D','K_H'),
            'meta'       => [
                'day'       => $dayOfWeek,
                'month'     => $month,
                'hourStart' => $hourStart,
                'peak50'    => $peak50,
            ],
        ];
    }

}
