import {defineStore} from 'pinia';
import {authService} from '@/api/authService';

const user = JSON.parse(localStorage.getItem('user') || 'null') || undefined;
const userPermissions = JSON.parse(localStorage.getItem('userPermissions') || 'null') || undefined;

export const useAuthStore = defineStore('auth/user', {
    state: () => ({
        user: user,
        isLoggedIn: !!user,
        userPermissions: userPermissions,
    }),
    actions: {
        async login(payload) {
            try {
                const response = await authService.login(payload);
                console.log(response);

                localStorage.setItem('user', JSON.stringify(response.data.data.user));
                localStorage.setItem('userPermissions', JSON.stringify(response.data.data.user.rolePermissions));
                localStorage.setItem('accessToken', response.data.data.accessToken);

                // Обновляем state внутри стора:
                this.user = response.data.data.user;
                this.userPermissions = response.data.data.user.rolePermissions;
                this.isLoggedIn = true;

                // Возвращение ответа в случае успеха
                return {response};
            } catch (e) {
                // console.error(e);
                return {error: e.response}; // Возвращение ошибки
            }
        },

        async register(payload) {
            try {
                const response = await authService.register(payload);

                // Возвращение ответа в случае успеха
                return {response};
            } catch (e) {
                // console.error(e);
                return {error: e.response}; // Возвращение ошибки
            }
        },
        async changePassword(payload) {
            try {
                const response = await authService.changePassword(payload);
                this.clearStore();
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        async logout() {
            try {
                await authService.logout().then(() => {
                    this.clearStore();
                });
            } catch (error) {
                if (error.response && error.response.status === 401) {
                    this.clearStore();
                }
            }
        },
        clearStore() {
            localStorage.removeItem('user');
            localStorage.removeItem('userPermissions');
            localStorage.removeItem('accessToken');
            this.user = null;
            this.userPermissions = null;
            this.isLoggedIn = false;
        },
        hasPermission(permissionsToCheck) {
            const permissionsArray = Array.isArray(permissionsToCheck) ? permissionsToCheck : [permissionsToCheck];
            return permissionsArray.some((permission) => this.userPermissions.includes(permission));
        },
        async getUsers() {
            try {
                const response = await authService.getUsers();
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        async getUserInfo(id) {
            try {
                const response = await authService.getUserInfo(id);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        async changeUserInfo(id, payload) {
            try {
                const response = await authService.changeUserInfo(id, payload);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        async deleteUser(id) {
            try {
                const response = await authService.deleteUser(id);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        async getLogs(page) {
            try {
                const response = await authService.getLogs(page);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        async getPointInfo(id) {
            try {
                const response = await authService.getPointInfo(id);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        async getPointInfoForEdit(pointId) {
            try {
                const {point} = await authStore.fetchPointForEdit(pointId.value)
                return {point};
            } catch (e) {
                return {error: e.point};
            }
        },
        async savePointPlan(payload) {
            try {
                const response = await authService.savePointPlan(payload);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },

        async editPoint(formData) {
            try {
                const response = await authService.editPoint(formData);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        async getSchedulDraftInfo(id) {
            try {
                const response = await authService.getSchedulDraftInfo(id);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        async createSignature(id) {
            try {
                const response = await authService.createSignature(id);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },

        async getSchedulesInfo(id) {
            try {
                const response = await authService.getSchedulesInfo(id);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        async saveOperatorPoint(payload) {
            try {
                const response = await authService.saveOperatorPoint(payload);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },

        async getRegionsAndSeniorManagers() {
            try {
                const response = await authService.getRegionsAndSeniorManagers();
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },

        async purposeSenManager(payload) {
            try {
                const response = await authService.purposeSenManager(payload);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },

        async getRegionsAndManagers(id) {
            try {
                const response = await authService.getRegionsAndManagers(id);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },

        async purposeManager(payload) {
            try {
                const response = await authService.purposeManager(payload);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },

        async getPointsAndOperators() {
            try {
                const response = await authService.getPointsAndOperators();
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },

        async proposeOperatorToPoint(payload) {
            try {
                const response = await authService.proposeOperatorToPoint(payload);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },

        async getPointForMap() {
            try {
                const response = await authService.getPointForMap();
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },

        async getPointList() {
            try {
                const response = await authService.getPointList();
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        async createPoint(payload) {
            try {
                const response = await authService.createPoint(payload);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        async infoForCreatePoint() {
            try {
                const {regions} = await authStore.fetchRegionsForSelect()
                return {regions};
            } catch (e) {
                return {error: e.regions};
            }
        },

        async getSchedulePlanList(id) {
            try {
                const response = await authService.getSchedulePlanList(id);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },

        async creteSchadulPlan(payload) {
            try {
                const response = await authService.creteSchadulPlan(payload);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },

        async schedulePlanInfo(id) {
            try {
                const response = await authService.schedulePlanInfo(id);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },

        async getPointPlanList(id) {
            try {
                const response = await authService.getPointPlanList(id);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        async getPointPlanInfo(id) {
            try {
                const response = await authService.getPointPlanInfo(id);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        async updatePointPlan(id, payload) {
            try {
                const response = await authService.updatePointPlan(id, payload);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },

        async getPointsOperator(params = {}) {
            try {
                const response = await authService.getPointsOperator(params);
                return {response};
            } catch (e) {
                return {error: e.response || e};
            }
        },

        async uploadFile(payload) {
            try {
                const response = await authService.uploadFile(payload);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        async deletePointPlan(id) {
            try {
                const response = await authService.deletePointPlan(id);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },

        async getNotifications() {
            try {
                const response = await authService.getNotifications();
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },

        async markNotificationAsRead(id) {
            try {
                const response = await authService.markNotificationAsRead(id);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        async generatePointPlans(payload) {
            this.loading = true
            this.error = null
            try {
                const res = await authService.generatePointPlans(payload)
                // возвращаем сразу результат: { rows_created: … }
                return {response: res.data}
            } catch (e) {
                // если сервер вернул JSON с ошибкой, возьмём из него message
                const errData = e.response?.data || {}
                return {error: errData}
            } finally {
                this.loading = false
            }
        },
        /**
         * Загрузить список заявок
         */
        async getRequests() {
            try {
                const response = await authService.getRequests();
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        /** Забираем регионы с актуальными group_manager-ами и списком кандидатов */
        async getRegionsAndGroupManagers() {
            try {
                const response = await authService.getRegionsAndGroupManagers();
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        /** Назначаем менеджера группы на регион */
        async assignGroupManager(payload) {
            // payload = { region_id, group_manager_id }
            try {
                const response = await authService.assignGroupManager(payload);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },

        // --------------- 2) Работа с subgroup_manager в районах ---------------

        /** Забираем районы с актуальными subgroup_manager-ами и списком кандидатов */
        async getDistrictsAndSubgroupManagers() {
            try {
                const response = await authService.getDistrictsAndSubgroupManagers();
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        /** Назначаем менеджера подгруппы на район */
        async assignSubgroupManager(payload) {
            try {
                const response = await authService.assignSubgroupManager(payload);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        // в actions:
        async getDistrictsForGroupAssignment() {
            try {
                const response = await authService.getDistrictsForGroupAssignment();
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        async assignGroupManagerToDistrict(payload) {
            try {
                const response = await authService.assignGroupManagerToDistrict(payload);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        async getSchedulePlanMeta(id) {
            try {
                const response = await authService.getSchedulePlanMeta(id);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },

        async getCalendarView(id, query = {}) {
            try {
                const response = await authService.getCalendarView(id, query)
                return {response}
            } catch (e) {
                return {error: e.response || e}
            }
        },

        /**
         * Подписать план-график по заявке на указанный месяц.
         * payload: { request_id: number, month: number, year: number }
         */
        async signPlan(payload) {
            try {
                const response = await authService.signPlan(payload);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },
        async getCounters(params = {}) {
            try {
                const res = await authService.getCounters(params)
                // Поскольку бекенд возвращает «сырая» JSON-массив, то:
                // res.data === [ {id:…, first_name:…, …}, { … }, … ]
                return {data: res.data, error: null}
            } catch (e) {
                console.error('Ошибка API getCounters:', e)
                return {data: [], error: e.response}
            }
        },
        // 2) Создать нового учётчика
        async createCounter(payload) {
            try {
                const res = await authService.createCounter(payload);
                return {response: res};
            } catch (e) {
                console.error('Ошибка при createCounter:', e);
                return {error: e.response};
            }
        },

        // 3) Обновить существующего учётчика
        async updateCounter(id, payload) {
            try {
                const res = await authService.updateCounter(id, payload);
                return {response: res};
            } catch (e) {
                console.error('Ошибка при updateCounter:', e);
                return {error: e.response};
            }
        },

        // 4) Удалить учётчика
        async deleteCounter(id) {
            try {
                const res = await authService.deleteCounter(id);
                return {response: res};
            } catch (e) {
                console.error('Ошибка при deleteCounter:', e);
                return {error: e.response};
            }
        },

        /** Загрузить подписанные месяцы для выбранной заявки */
        async getSignedPeriods(requestId) {
            try {
                const response = await authService.getSignedPeriods(requestId);
                return {response};
            } catch (e) {
                return {error: e.response || e};
            }
        },
        async fetchTrafficReport(range) {
            try {
                const {response} = await authService.getTrafficReport(range);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },

        async downloadTrafficReport(range) {
            await authService.exportTrafficReport(range);
        },

        async fetchDetailedTrafficReport(params) {
            try {
                const response = await authService.getTrafficDetailedReport(params)
                return {response}
            } catch (e) {
                return {error: e.response}
            }
        },

        async fetchPointsForMap(districtId = null) {
            try {
                // сразу получаем axios-ответ
                const res = await authService.getPointForMap(districtId);
                return {points: res.data.points};
            } catch (e) {
                console.error('Ошибка при загрузке точек для карты:', e);
                return {error: e.response};
            }
        },

        async downloadDetailedTrafficReport(params, districtName) {
            await authService.exportTrafficDetailedReport(params, districtName)
        },

        async downloadScheduleExcel(id, opts) {
            // opts = { month, district, freq, request_id }
            await authService.exportScheduleExcel(id, opts);
        },

        async getRefManagers(level) {
            try {
                const response = await authService.getRefManagers(level)
                return {response}
            } catch (e) {
                return {error: e.response}
            }
        },
        async getRefDistricts() {
            try {
                const response = await authService.getRefDistricts()
                return {response}
            } catch (e) {
                return {error: e.response}
            }
        },

        /** для выпадашки регионов и районов */
        async fetchRegionsForSelect() {
            try {
                const res = await authService.getRegions();
                // тут res.data = { data: [ ... ] }
                return {regions: res.data.data};
            } catch (e) {
                console.error('Ошибка при загрузке регионов:', e);
                return {regions: [], error: e.response};
            }
        },

        /** детальная инфа по одной точке */
        async fetchPointForEdit(id) {
            try {
                const res = await authService.getPoint(id);
                return {point: res.data.data};
            } catch (e) {
                console.error('Ошибка при загрузке точки:', e);
                return {point: null, error: e.response};
            }
        },
        async saveEditedPoint(id, formData) {
            try {
                const res = await authService.updatePoint(id, formData);
                return {response: res};
            } catch (e) {
                return {error: e.response};
            }
        },

        async getPointNames() {
            try {
                const response = await authService.getPointNames()
                return {response}
            } catch (e) {
                return {error: e.response}
            }
        },

        /** список объектов логов для фильтра «Объект» */
        async fetchLogObjects() {
            try {
                const res = await authService.getLogObjects();
                // res.data = { objects: [...] }
                return {objects: res.data.objects};
            } catch (e) {
                console.error('Ошибка при загрузке объектов логов:', e);
                return {objects: [], error: e.response};
            }
        },

        async createRequest(payloads) {
            this.loading = true
            try {
                const res = await authService.createRequest(payloads)
                // возвращаем сразу тело ответа
                return {response: res.data}
            } catch (e) {
                this.error = e.response
                return {error: e.response}
            } finally {
                this.loading = false
            }
        },

        async addPointsToRequest(id, pointIds) {
            try {
                const res = await authService.addPointsToRequest(id, pointIds);
                return {objects: res.data.objects};
            } catch (e) {
                console.error('Ошибка при добавлении точек в заявку:', e);
                return {objects: [], error: e.response};
            }
        },
        async getPointsByRequest(id) {
            try {
                const res = await authService.getPointsByRequest(id)
                return {response: res}
            } catch (e) {
                console.error('Ошибка fetch PointsByRequest:', e)
                return {error: e.response}
            }
        },
        /** Обновить данные заявки */
        async changeRequest(id, payload) {
            this.loading = true
            this.error = null
            try {
                const res = await authService.changeRequest(id, payload)
                return {response: res.data}
            } catch (e) {
                this.error = e.response
                return {error: e.response}
            } finally {
                this.loading = false
            }
        },

        async downloadKc2Template(params, districtName) {
            try {
                await authService.downloadKc2Template(params, districtName);
            } catch (e) {
                console.error('Не удалось скачать акт КС-2:', e);
                // можно бросить или вернуть ошибку
                return {error: e.response};
            }
        },


        /**
         * Получить ссылки на фото для отчёта
         * @param {number} id — ID заявки
         * @param {number|null} districtId
         * @param {number} sheets — кол-во листов
         */
        async getRequestPhotos(id, districtId = null, sheets = 1) {
            this.loading = true;
            try {
                const res = await authService.getRequestPhotos(id, districtId, sheets);
                // res.data — массив фото
                return {response: res.data};
            } catch (e) {
                console.error('Ошибка загрузки фото отчёта:', e);
                return {error: e.response};
            } finally {
                this.loading = false;
            }
        },

        /**
         * Импорт точек по именам (из Excel/CSV)
         */
        async importPointsByName(requestId, file) {
            try {
                const response = await authService.importPointsByName(requestId, file);
                return {response: response.data, error: null};
            } catch (err) {
                return {response: null, error: err.response || err};
            }
        },

        /**
         * Синхронизация точек: вызываем sync-points, деактивируем работы
         */
        async syncPoints(requestId, pointIds) {
            try {
                const response = await authService.syncPoints(requestId, pointIds);
                return {response: response.data, error: null};
            } catch (err) {
                return {response: null, error: err.response || err};
            }
        },
        async lockRequest(requestId) {
            try {
                const response = await authService.lockRequest(requestId);
                return {response};
            } catch (e) {
                return {error: e.response};
            }
        },

        /** Получить замеры за предыдущий год/месяц для автозаполнения */
        async getPreviousStats(point_id, year, month) {
            try {
                const response = await authService.getPreviousStats(point_id, year, month)
                // возвращаем { response }, чтобы фронт мог различать success/error
                return {response}
            } catch (e) {
                return {error: e.response}
            }
        },

        async fetchRequestDateRange(request_id) {
            try {
                const response = await authService.getRequestDateRange(request_id)
                // возвращаем { response }, чтобы фронт мог различать success/error
                return {response}
            } catch (e) {
                return {error: e.response}
            }
        },
        async fetchRequestDistrictRanges(request_id) {
            try {
                const response = await authService.getRequestDistrictRanges(request_id);
                return { response };
            } catch (e) {
                return { error: e.response };
            }
        },

        async startPhotoReport(requestId, districtId) {
            try {
                const response = await authService.startPhotoReport(requestId, districtId);
                return { response: response.data, error: null };
            } catch (err) {
                return { response: null, error: err.response || err };
            }
        },

        async getPhotoReportStatus(jobId) {
            try {
                const response = await authService.getPhotoReportStatus(jobId);
                return { response: response.data, error: null };
            } catch (err) {
                return { response: null, error: err.response || err };
            }
        },

        /**
         * Получить список всех фото-отчётов текущего пользователя
         */
        async listPhotoReports() {
            try {
                const response = await authService.listPhotoReports()
                // response.data — массив { jobId, requestName, districtName, createdAt, status, url, error }
                return { response: response.data, error: null }
            } catch (e) {
                console.error('Не удалось загрузить список фото-отчетов:', e)
                return { response: null, error: e.response || e }
            }
        },

    },
});
